package ru.yandex.calendar.util.validation;

import java.util.regex.Pattern;

import org.joda.time.DateTimeZone;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.frontend.web.cmd.run.ParameterValidationException;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.misc.email.Emails;
import ru.yandex.misc.lang.StringUtils;

/**
 * Performs request parameter validations
 * @author ssytnik
 */
public class RequestValidator {

    public static final String URL_PATTERN_STR = "(?:(?:https?|webcal)://)?[\\w-\\.\\?\\+%&=/]+";
    public static final Pattern URL_PATTERN = Pattern.compile(URL_PATTERN_STR);

    public static void addInvalidParameterCode(String paramName, String paramValue) {
        throw new ParameterValidationException("invalid parameter " + paramName + " value " + paramValue);
    }

    private static void addNoParameterCode(String paramName) {
        throw new ParameterValidationException("required parameter " + paramName + " missing");
    }

    public static void validate(IAtomicValidator v, boolean isRequired, String name, Object value) {
        final String valueStr = value == null ? null : value.toString();
        if (isRequired) {
            if (StringUtils.isEmpty(valueStr)) {
                RequestValidator.addNoParameterCode(name);
            }
        }
        if (v != null && StringUtils.isNotEmpty(valueStr)) {
            v.validate(name, valueStr);
        }
    }

    public static void validate(IAtomicArrayValidator v, String name, String values) {
        ListF<String> result;
        if (StringUtils.isNotEmpty(values))
            result = Cf.list(values.split(","));
        else
            result = Cf.list();
        v.validate(name, result);
    }

    public interface IAtomicValidator {
        public void validate(String name, String value);
    }

//    public RequestValidator validateOptional(IAtomicValidator v, String... names) {
//        return validate(v, false, names);
//    }
//    public RequestValidator validateRequired(IAtomicValidator v, String... names) {
//        return validate(v, true, names);
//    }
//    public RequestValidator validate(IAtomicValidator v, boolean isRequired, String... names) {
//        for (String name : names) { validate(v, isRequired, name, rw.getNullableString(name)); }
//        return this;
//    }

    public static void validateOptional(IAtomicValidator v, String name, Object value) {
        RequestValidator.validate(v, false, name, value);
    }

    public static void validateRequired(IAtomicValidator v, String name,
            Object value) {
        RequestValidator.validate(v, true, name, value);
    }

    public interface IAtomicArrayValidator {
        public void validate(String name, ListF<String> values);
    }

//    public RequestValidator validate(IAtomicArrayValidator v, String name) {
//        v.validate(this, name, rw.getArray(name));
//        return this;
//    }

    public static final IAtomicValidator DATE = new IAtomicValidator() {
        public void validate(String name, String value) {
            try {
                DateTimeFormatter.toNullableDateUnsafe(value);
            } catch (Exception e) {
                RequestValidator.addInvalidParameterCode(name, value);
            }
        }
    };

    public static final IAtomicValidator EMAIL = new IAtomicValidator() {
        @Override public void validate(String name, String value) {
            if (!Emails.isValid(value)) { RequestValidator.addInvalidParameterCode(name, value); }
        }
    };

    public static final IAtomicValidator URL = new IAtomicValidator() {
        @Override public void validate(String name, String value) {
            if (!URL_PATTERN.matcher(value).matches()) { RequestValidator.addInvalidParameterCode(name, value); }
        }
    };

    public static final IAtomicValidator LONG = new IAtomicValidator() {
        @Override public void validate(String name, String value) {
            try {
                Long.parseLong(value);
            } catch (Exception e) {
                RequestValidator.addInvalidParameterCode(name, value);
            }
        }
    };

    public static final IAtomicValidator NON_NEGATIVE = new IAtomicValidator() {
        @Override public void validate(String name, String value) {
            try {
                long v = Long.parseLong(value);
                if (v < 0) { RequestValidator.addInvalidParameterCode(name, value); }
            } catch (Exception e) {
                RequestValidator.addInvalidParameterCode(name, value);
            }
        }
    };

    public static final IAtomicValidator POSITIVE = new IAtomicValidator() {
        @Override public void validate(String name, String value) {
            try {
                long v = Long.parseLong(value);
                if (v <= 0) { RequestValidator.addInvalidParameterCode(name, value); }
            } catch (Exception e) {
                RequestValidator.addInvalidParameterCode(name, value);
            }
        }
    };

    public static final IAtomicValidator TIMESTAMP = new IAtomicValidator() {
        @Override public void validate(String name, String value) {
            try {
                DateTimeFormatter.toNullableTimestampUnsafe(value, DateTimeZone.UTC);
            } catch (Exception e) {
                RequestValidator.addInvalidParameterCode(name, value);
            }
        }
    };

    // either timestamp, or P{period}[{+|-}] (period with optional floor/ceiling alignment)
    // like 'P1D+' == 'add one day to the (well-known) start-ts and then align days up',
    // or L{local time, hours}H, or "AUTO" (end-ts is chosen by the calendar)
    public static final IAtomicValidator EXT_TIMESTAMP = new IAtomicValidator() {
        @Override public void validate(String name, String value) {
            if ("AUTO".equalsIgnoreCase(value)) { return; } // ok (auto / chosen by calendar)
            if (AuxDateTime.toLocalTimeNullable(value) != null) { return; } // ok (local time)
            if (AuxDateTime.toPeriodAndRoundingNullable(value) != null) { return; } // ok (period)
            TIMESTAMP.validate(name, value); // try timestamp -- delegate
        }
    };


    public static final IAtomicArrayValidator LONG_ARRAY = new IAtomicArrayValidator() {
        public void validate(String name, ListF<String> values) {
            try {
                AuxColl.toLongArray(values);
            } catch (Exception e) {
                RequestValidator.addInvalidParameterCode(name, values.toString());
            }
        }
    };
}
