package ru.yandex.calendar.util.wiki;

import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.lang.Validate;

/**
 * @author Eugene Voytitsky
 */
public abstract class AuthMarker {
    // WIKI API doc declares that auth marker is valid during 2 hours
    // reduce 2 hours by 5 minutes for safety
    public static final long TIME_TO_LIVE_MS = (2 * 60 - 5) * 60 * 1000;

    public static AuthMarker cons(String marker) {
        return new AuthMarkerImpl(marker, System.currentTimeMillis());
    }

    public static AuthMarker expired() {
        return new ExpiredAuthMarkerImpl();
    }

    public abstract void used(String usedMarker);
    public abstract String getMarker();
    public abstract boolean isValid();

    protected static class AuthMarkerImpl extends AuthMarker {
        private final String marker;
        private final AtomicLong lastUsedTs;

        private AuthMarkerImpl(String marker, long lastUsedTs) {
            Validate.notNull(marker);
            this.marker = marker;
            this.lastUsedTs = new AtomicLong(lastUsedTs);
        }

        @Override
        public void used(String marker) {
            if (this.marker.equals(marker)) {
                lastUsedTs.set(System.currentTimeMillis());
            }
        }

        @Override
        public String getMarker() {
            return marker;
        }

        @Override
        public boolean isValid() {
            return System.currentTimeMillis() - lastUsedTs.get() < TIME_TO_LIVE_MS;
        }

    }

    protected static class ExpiredAuthMarkerImpl extends AuthMarker {
        @Override
        public void used(String usedMarker) {
            throw new UnsupportedOperationException();
        }

        @Override
        public String getMarker() {
            throw new UnsupportedOperationException();
        }

        @Override
        public boolean isValid() {
            return false;
        }
    }
}
