package ru.yandex.calendar.util.xml;

import java.util.Collection;

import javax.annotation.Nullable;

import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.ReadableInstant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.event.EventDateTime;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.enums.EnumUtils;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.xml.XmlUtils;

/**
 * @author ssytnik
 */
public class CalendarXmlizer {

    /**
     * Converts Object to String with appropriate formatting
     * @param value value of any type, can be null
     * @return string representation, can be null\
     * @throws CommandRunException if date time formatter is null when value is java.util.Date,
     * or other underlying error occurs
     */
    public static String objToXmlStr(Object value, DateTimeZone tz) {
        if (value != null) {
            if (value instanceof Boolean) {
                return Binary.toString((Boolean) value);
            } else if (value instanceof LocalTime) {
                return DateTimeFormatter.formatLocalTimeForMachines((LocalTime) value);
            } else if (value instanceof LocalDate) {
                return DateTimeFormatter.formatLocalDateForMachines((LocalDate) value);
            } else if (value instanceof LocalDateTime) {
                return DateTimeFormatter.formatLocalDateTimeForMachines((LocalDateTime) value);
            } else if (value instanceof ReadableInstant) {
                return DateTimeFormatter.formatForMachines((ReadableInstant) value, tz);
            } else if (value instanceof java.sql.Date) {
                return DateTimeFormatter.formatLocalDateForMachines((java.sql.Date) value, tz);
            } else if (value instanceof java.util.Date) {
                return DateTimeFormatter.formatForMachines(((java.util.Date) value).getTime(), tz);
            } else if (value instanceof EventDateTime) {
                return DateTimeFormatter.formatForMachines((EventDateTime) value, tz);
            } else if (value instanceof String) {
                return (String) value;
            } else if (value instanceof StringEnum) {
                return ((StringEnum) value).value();
            } if (value instanceof Enum) {
                return EnumUtils.toXmlName((Enum) value);
            } if (value instanceof Email) {
                return Emails.getUnicoded((Email) value);
            } else {
                return value.toString();
            }
        } else {
            return null;
        }
    }

    private static String invalidCharactersToSpacesNullable(String string) {
        if (string != null)
            return XmlUtils.invalidCharsToSpaces(string).toString();
        else
            return null;
    }

    /**
     * For given parent element, adds a new content element with given name and value
     * @param eParent parent element
     * @param name name of the child element
     * @param value value of the child element to be formatted as text
     * @return created child element
     */
    public static Element appendDtfElm(
            Element eParent, String name, Object value, DateTimeZone tz)
    {

        String xmlName = CamelWords.parse(name).toXmlName();
        String xmlValue = invalidCharactersToSpacesNullable(objToXmlStr(value, tz));
        Element child = new Element(xmlName);
        child.setText(xmlValue);
        eParent.addContent(child);
        return child;
    }

    /**
     * The same as {@link #appendDtfElm(Element, String, Object, DateTimeZone)},
     * but does not accept dtf parameter. The programmer is responsible of
     * choosing between these routines.
     * @param value value of the child element to be formatted as text
     * @return parent element with new child element added
     */
    public static Element appendElm(Element eParent, String name, Object value, DateTimeZone tz) {
        return appendDtfElm(eParent, name, value, tz);
    }

    public static Element appendElm(Element eParent, String name) {
        return appendElm(eParent, name, null, null);
    }

    public static void appendElm(Element eParent, String name, String value) {
        appendElm(eParent, name, value, null);
    }

    public static void appendElm(Element eParent, String name, Email value) {
        appendElm(eParent, name, Emails.getUnicoded(value));
    }

    public static void appendElm(Element eParent, String name, Number value) {
        appendElm(eParent, name, value, null);
    }

    public static void appendElm(Element eParent, String name, Boolean value) {
        appendElm(eParent, name, value, null);
    }

    public static void appendElm(Element eParent, String name, LocalDateTime value) {
        appendElm(eParent, name, value, null);
    }

    public static void appendElm(Element eParent, String name, Enum value) {
        appendElm(eParent, name, value, null);
    }

    /**
     * Adds an array
     * @param <T>                  array objects class (just for better consistency)
     * @param collName collection element name (if null, then 'itemName's used)
     * @param itemName single item element name
     * @param values single item values
     * @return parent element
     */
    public static <T> Element appendElmColl(
            Element eParent, @Nullable String collName, String itemName, Collection<T> values)
    {
        return eParent.addContent(createElmColl(collName, itemName, values));
    }

    public static <T> Element createElmColl(
            @Nullable String collName, String itemName, Collection<T> values)
    {
        String eCollName = CamelWords.parse((StringUtils.isNotEmpty(collName) ? collName : itemName + "s")).toXmlName();
        Element eColl = new Element(eCollName);
        for (T value : values) {
            appendElm(eColl, itemName, value, null);
        }
        return eColl;
    }

    public static <T> void appendElmArray(
            Element eParent, @Nullable String collName, String itemName, T[] values)
    {
        appendElmColl(eParent, collName, itemName, Cf.x(values));
    }

    // Sets a date/time attribute with given name and value. See appendElm() for details.
    // NOTE: value cannot be null for attribute.
    public static Element setDtfAttr(Element eParent, String name, Object value, DateTimeZone dateTimeZone)
    {

        String xmlName = CamelWords.parse(name).toXmlName();
        String xmlValue = objToXmlStr(value, dateTimeZone);
        return eParent.setAttribute(xmlName, xmlValue != null ? xmlValue : "null");
    }

    // NOTE: value cannot be null for attribute
    public static Element setAttr(Element eParent, String name, Object value, DateTimeZone tz) {
        return setDtfAttr(eParent, name, value, tz);
    }

    public static void setAttr(Element eParent, String name, String value) {
        setAttr(eParent, name, value, null);
    }

    public static Element setAttr(Element eParent, String name, Number value) {
        return setAttr(eParent, name, value, null);
    }

    public static void setAttr(Element eParent, String name, Boolean value) {
        setAttr(eParent, name, value, null);
    }

    public static void setAttr(Element eParent, String name, LocalDate value) {
        setAttr(eParent, name, value, null);
    }

    public static void setAttr(Element eParent, String name, LocalTime value) {
        setAttr(eParent, name, value, null);
    }

    public static void setAttr(Element eParent, String name, LocalDateTime value) {
        setAttr(eParent, name, value, null);
    }

    // Sets a date/time text with given value.
    public static Element setDtfText(Element eParent, Object value, DateTimeZone tz) {
        return eParent.setText(objToXmlStr(value, tz));
    }

    // Sets a text with given value.
    public static Element setText(Element eParent, Object value, DateTimeZone tz) {
        return setDtfText(eParent, value, tz);
    }

    public static void setText(Element eParent, Long value) {
        setText(eParent, value, null);
    }

} //~
