package ru.yandex.calendar.util.xml;

import java.util.regex.Pattern;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import org.jdom.Content;
import org.jdom.Element;
import org.jdom.Text;
import org.springframework.web.util.HtmlUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.enums.EnumResolver;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.regex.Matcher2;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.xml.XmlUtils;
import ru.yandex.misc.xml.jdom.JdomUtils;

/**
 * Contains information about how to do replacement
 * from text pieces to xml tags with attributes
 * @author ssytnik
 */
public enum TagReplacement {
    A, B, P;

    public static final EnumResolver<TagReplacement> R = EnumResolver.er(TagReplacement.class);

    /** Useful for {@link Action#TEXT_STRIP} mode */
    private interface TagToTextConverter { String asText(Element e); }

    /** Useful to guard bad attribute values, e.g. see CAL-2089 */
    private interface AttributeValidator { boolean isValidAttribute(String aName, String value); }

    /** Specifies what to do (how to process) given text */
    public static enum Action {
        /** convert tag text pieces to xml tags and known attributes */
        CONVERT_XML,
        /** strip tags from the text */
        TEXT_STRIP
    }

    private static final String Q = "[\"]", NOT_Q = "[^\"]", NOT_QS = NOT_Q + "+?";
    private static final String LETTER = "[a-z]", SPACE = "\\s";
    private static final String LETTERS = LETTER + '+', SPACES = SPACE + '+', SPACES_O = SPACE + '*';

    private static final String ATTR = "(?:" + SPACES + '(' + LETTERS + ')' + "=" + Q + "(" + NOT_QS + ")" + Q + ")";
    private static final String TAG_NAME_WITH_ATTRS = "(" + LETTER + "+)(" + ATTR + "*)" + SPACES_O;
    /**
     * We are attempting to use simple regex pattern instead of proper grammar.
     * In its 2nd part (full tag syntax), we allow everything between opening
     * and closing tags, and that's result in the following inconsistencies:
     * not supported (work incorrectly)
     * - nested tags of the same type;
     * - closing tags in attribute values.
     */
    private static final Pattern2 TAG_PATTERN = new Pattern2(Pattern.compile(
        "(?:(<" + TAG_NAME_WITH_ATTRS + "/>))" +
        "|(?:(<" + TAG_NAME_WITH_ATTRS + ">)((?s:.*?))</\\7>)",
        Pattern.CASE_INSENSITIVE
    ));
    private static final Pattern2 ATTR_PATTERN = Pattern2.compile(ATTR, Pattern.CASE_INSENSITIVE);

    private SetF<String> reqAttrSet = Cf.set(); // required known attributes
    private SetF<TagReplacement> allowedTagSet = Cf.set(); // possible nested tag types

    private Option<TagToTextConverter> toTextConverter = Option.empty();
    private Option<AttributeValidator> attrValidator = Option.empty();

    private TagReplacement setReqAttrs(String... reqAttrs) {
        this.reqAttrSet = Cf.hashSet(reqAttrs);
        return this;
    }

    private void setAllowedTags(TagReplacement... allowedTags) {
        this.allowedTagSet = Cf.hashSet(allowedTags);
    }

    private void setToTextConverter(TagToTextConverter toTextConverter) {
        this.toTextConverter = Option.of(toTextConverter);
    }

    private void setAttrValidator(AttributeValidator attrValidator) {
        this.attrValidator = Option.of(attrValidator);
    }

    static {
        A.setReqAttrs("href").setAllowedTags(B);
        A.setToTextConverter(new TagToTextConverter() {
            public String asText(Element e) {
                String text = e.getText(), href = e.getAttributeValue("href");
                if (text.equals(href)) { return href; } // return link
                return text + " (" + href + ")"; // return text and link
            }
        });
        A.setAttrValidator(new AttributeValidator() {
            public boolean isValidAttribute(String name, String value) {
                return !name.equals("href") || value.startsWith("http") || value.startsWith("www");
            }
        });
        B.setReqAttrs().setAllowedTags(A);
        P.setReqAttrs().setAllowedTags(A, B);
    }

    private String asText(Element e) {
        return !toTextConverter.isPresent() ? e.getText() : toTextConverter.get().asText(e);
    }

    private boolean isValidAttribute(String name, String value) {
        return !attrValidator.isPresent() || attrValidator.get().isValidAttribute(name, value);
    }

    private Function2B<String, String> isValidAttributeF() {
        return new Function2B<String, String>() {
            public boolean apply(String name, String value) {
                return isValidAttribute(name, value);
            }
        };
    }

    private static Tuple2List<String, String> parseAttributes(String text) {
        Tuple2List<String, String> result = Tuple2List.arrayList();
        Matcher2 m = ATTR_PATTERN.matcher2(text);

        while (m.find()) {
            result.add(m.group(1).get(), m.group(2).get());
        }
        return result;
    }

    private static ListF<Content> replaceWithBrs(String text) {
        ListF<Content> result = Cf.arrayList();
        int pos = text.indexOf('\n');

        while (pos >= 0) {
            result.add(new Text(text.substring(0, pos)));
            result.add(new Element("br"));

            text = text.substring(pos + 1);
            pos = text.indexOf('\n');
        }
        result.add(new Text(text));
        return result;
    }

    private static ListF<Content> processTextNode(String text, Action action, Option<String> linkSignKey) {
        if (action == Action.CONVERT_XML) {
            ListF<Content> result = Cf.arrayList();
            for (Content content : replaceWithBrs(text)) {
                if (content instanceof Text) {
                    for (Content link : LinkReplacement.replace(((Text) content).getText())) {
                        Option<Element> element = Option.of(link).filterByType(Element.class).singleO();
                        Option<String> href = element.map(e -> e.getAttributeValue("href")).singleO();

                        if (href.isPresent() && !href.get().startsWith("mailto:") && linkSignKey.isPresent()) {
                            signLink(element.get(), linkSignKey.get());
                        }
                        result.add(link);
                    }
                } else {
                    result.add(content);
                }
            }
            return result;
        } else {
            return Cf.list(new Text(text));
        }
    }

    // text can be unset
    private static Element processTextInner(
            Element eParent, String text, Action action, SetF<TagReplacement> allowedTags, Option<String> linkSignKey)
    {
        text = XmlUtils.invalidCharsToSpaces(text).toString();

        if (StringUtils.isEmpty(text)) { return eParent; }

        Matcher2 m = TAG_PATTERN.matcher2(text);
        int pos = 0;
        String spacePrefix = ""; /* in order to insert a space after converted-AS_TEXT-<p>-tag */

        while (m.find()) {
            eParent.addContent(processTextNode(text.substring(pos, m.start()), action, linkSignKey));

            String tagName = m.group(2).orElse(m.group(7)).get();
            String attrsStr = m.group(3).orElse(m.group(8)).get();

            String beforeContent = m.group(1).orElse(m.group(6)).get();
            Option<String> content = m.group(11);

            pos = m.end();

            TagReplacement tag = TagReplacement.R.valueOfO(tagName).getOrNull();
            Tuple2List<String, String> attrs = parseAttributes(attrsStr);

            if (tag == null
                || !allowedTags.containsTs(tag)
                || !tag.reqAttrSet.forAll(attrs.get1().containsF())
                || !attrs.filterBy1(tag.reqAttrSet.containsF()).forAll(tag.isValidAttributeF()))
            {
                eParent.addContent(processTextNode(beforeContent, action, linkSignKey));
                return processTextInner(eParent,
                        text.substring(beforeContent.length() + m.start()), action, allowedTags, linkSignKey);
            }

            Element eTag = new Element(tagName);
            for (Tuple2<String, String> attr : attrs.filterBy1(tag.reqAttrSet.containsF())) {
                if (attr.get1().equals("href")) {
                    eTag.setAttribute(attr.get1(), LinkReplacement.toAsciiUrlWithSchema(attr.get2()));
                } else {
                    eTag.setAttribute(attr.get1(), attr.get2());
                }
            }
            if (tag == TagReplacement.A && linkSignKey.isPresent()) {
                signLink(eTag, linkSignKey.get());
            }

            if (content.isPresent()) {
                processTextInner(eTag, content.get(), action, allowedTags.intersect(tag.allowedTagSet), linkSignKey);
            }
            switch (action) {
                case CONVERT_XML: eParent.addContent(eTag); break;
                case TEXT_STRIP: eParent.addContent(spacePrefix + tag.asText(eTag)); break;
                default: throw new CommandRunException("Unexpected action: " + action);
            }
            spacePrefix = (P.equals(tag) ? " " : "");
        }
        return eParent.addContent(processTextNode(text.substring(pos), action, linkSignKey));
    }

    /** @see verstka /away.xml */
    private static void signLink(Element anchor, String key) {
        String url = anchor.getAttributeValue("href");
        String client = StringUtils.substringBefore(key, ":");

        anchor.setAttribute("data-sign", client.length() > 1
                ? hmacMd5("client=" + client + "&url=" + url + client, StringUtils.substringAfter(key, ":"))
                : key.substring(0, 1) + "," + Md5.A.digest(key + ":" + url).hex());
    }

    private static String hmacMd5(String message, String key) {
        try {
            Mac mac = Mac.getInstance("HmacMD5");
            mac.init(new SecretKeySpec(key.getBytes(), "HmacMD5"));

            return Hex.encode(mac.doFinal(message.getBytes()));

        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * At given text, recognizes known or unknown tags and processes conversion as requested
     * @param resTagName result tag name
     * @param text given text (can be unset)
     * @param action see {@link Action}
     * NOTE: returned element does not contain any text pieces quoted.
     * @param allowedTags specifies tags that are allowed to be converted to xml
     * @return xml element
     */
    public static Element processText(
            String resTagName, String text, Action action, Option<String> linkSignKey, TagReplacement... allowedTags) {
        return processTextInner(new Element(resTagName), text, action, Cf.set(allowedTags), linkSignKey);
    }

    public static Element processText(String resTagName, String text, Action action, TagReplacement... allowedTags) {
        return processText(resTagName, text, action, Option.<String>empty(), allowedTags);
    }

    // Calls processText(), giving it full set of tags available
    public static Element processTextAllTags(String resTagName, String text, Action action) {
        return processText(resTagName, text, action, values());
    }

    public static String processText(
            String text, Action action, Option<String> linkSignKey, TagReplacement... allowedTags)
    {
        ListF<Content> contents = Cf.x(processText("xml", text, action, linkSignKey, allowedTags).getContent());

        StringBuilder sb = new StringBuilder();
        for (Content c : contents) {
            if (c instanceof Element) {
                sb.append(JdomUtils.I.writeElementToString((Element) c));
            } else if (c instanceof Text) {
                sb.append(HtmlUtils.htmlEscape(((Text) c).getText()));
            } else {
                throw new IllegalStateException("Unexpected content type " + c.getClass());
            }
        }
        return sb.toString();
    }

    public static String processText(String text, Action action, TagReplacement... allowedTags) {
        return processText(text, action, Option.<String>empty(), allowedTags);
    }
}
