package ru.yandex.calendar.util.xml;

import javax.xml.transform.Source;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamSource;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.transform.JDOMResult;
import org.jdom.transform.JDOMSource;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.util.resources.CalendarClassPathResourceUtils.ResourceName;
import ru.yandex.calendar.util.resources.ResourceManagerURIResolver;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.xml.XmlException;
import ru.yandex.misc.xml.XmlUtils;
import ru.yandex.misc.xml.saxon.SaxonUtils;

/**
 * @author Stepan Koltsov
 * @author ssytnik
 */
public class XslUtils {
    private static final Logger logger = LoggerFactory.getLogger(XslUtils.class);

    private static final MapF<String, TransformerFactory> xslTransformerFactories;

    static {
        xslTransformerFactories = Cf.hashMap();
        for (String language : Language.R.valuesList().map(Language::value)) {
            TransformerFactory xslTransformerFactory = SaxonUtils.newTransformerFactory();
            xslTransformerFactory.setURIResolver(new ResourceManagerURIResolver(language));
            xslTransformerFactories.put(language, xslTransformerFactory);
        }
    }

    private static final MapF<String, MapF<String, Templates>> prepared = Cf.concurrentHashMap();

    public static void prepareMailXsls() {
        Cf.x(ResourceName.values()).forEach(res ->
                prepared.put(res.getSS().getPublicId(), res.getSupportedLangs().map(Language::value)
                        .toMap(l -> l, l -> SaxonUtils.newTemplates(res.getIS(), new ResourceManagerURIResolver(l)))));
    }

    /**
     * Applies xsl that can be read from the given source, to the given xml element
     * @return transformation result
     */
    public static Document applyXslt(Element eXml, Source srcXsl) {
        Validate.notNull(eXml);
        try {
            Source srcXml = new JDOMSource(eXml);
            String language = Option.ofNullable(eXml.getChildText("language")).getOrElse("ru");
            Transformer transformer = newTransformer(srcXsl, language);
            JDOMResult result = new JDOMResult();
            transformer.transform(srcXml, result);
            return result.getDocument();
        } catch (Exception e) {
            throw XmlUtils.translate(e);
        }
    }

    private static Transformer newTransformer(Source srcXsl, String language) {
        Validate.notNull(srcXsl);

        Option<MapF<String, Templates>> templates = prepared.getO(
                srcXsl instanceof StreamSource ? ((StreamSource) srcXsl).getPublicId() : "");

        if (templates.isPresent()) {
            String lang = templates.get().containsKeyTs(language) ? language : "ru";
            try {
                return templates.get().getOrThrow(lang).newTransformer();

            } catch (TransformerConfigurationException e) {
                throw XmlUtils.translate(e);
            }
        }
        try {
            String lang = xslTransformerFactories.containsKeyTs(language) ? language : "ru";
            logger.debug("Creating xml transformer for {} language", lang);

            return xslTransformerFactories.getOrThrow(lang).newTransformer(srcXsl);

        } catch (Exception e) {
            throw new XmlException("failed to initialize transformer from " + XmlUtils.toString(srcXsl) + ": " + e, e);
        }
    }

} //~
