package ru.yandex.calendar.util.xmlorjson;

import ru.yandex.commune.json.write.JsonWriter;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.xml.stream.XmlWriter;

/**
 * @author gutman
 * @author Sergey Shinderuk
 */
public class JsonBuilder extends XmlOrJsonWriter {

    private final JsonWriter jw;
    private ObjectOrArrayWriter w = new RootWriter();

    public JsonBuilder(JsonWriter jw) {
        this.jw = jw;
    }

    @Override
    public void startObject(String name) {
        w.writeMemberStart(name);
        w = new ObjectWriter(w);
        w.writeStart();
    }

    @Override
    public void endObject() {
        Validate.isTrue(w instanceof ObjectWriter);
        w.writeEnd();
        w = w.getParent();
    }

    @Override
    public void startArray(String name) {
        w.writeMemberStart(name);
        w = new ArrayWriter(w);
        w.writeStart();
    }

    @Override
    public void endArray() {
        Validate.isTrue(w instanceof ArrayWriter);
        w.writeEnd();
        w = w.getParent();
    }

    @Override
    public void addTextField(String name, String value) {
        w.writeTextMember(name, value);
    }

    @Override
    public void addNumberField(String name, int value) {
        w.writeNumberMember(name, value);
    }

    @Override
    public void addNumberField(String name, long value) {
        w.writeNumberMember(name, value);
    }

    @Override
    public void addNumberField(String name, float value) {
        w.writeNumberMember(name, value);
    }

    @Override
    public void addNumberField(String name, double value) {
        w.writeNumberMember(name, value);
    }

    @Override
    public void addBooleanField(String name, boolean value) {
        w.writeBooleanMember(name, value);
    }

    @Override
    public void addNullField(String name) {
        w.writeNullMember(name);
    }

    @Override
    public boolean isXml() {
        return false;
    }

    @Override
    public JsonWriter getJsonWriter() {
        return jw;
    }

    @Override
    public XmlWriter getXmlWriter() {
        throw new UnsupportedOperationException();
    }

    private interface ObjectOrArrayWriter {
        ObjectOrArrayWriter getParent();

        void writeMemberStart(String name);
        void writeStart();
        void writeEnd();
        void writeTextMember(String name, String value);
        void writeNumberMember(String name, int value);
        void writeNumberMember(String name, long value);
        void writeNumberMember(String name, float value);
        void writeNumberMember(String name, double value);
        void writeBooleanMember(String name, boolean value);
        void writeNullMember(String name);
    }

    private abstract class AbstractJsonWriter implements ObjectOrArrayWriter {
        @Override
        public void writeTextMember(String name, String value) {
            this.writeMemberStart(name);
            JsonBuilder.this.jw.writeString(value);
        }

        @Override
        public void writeNumberMember(String name, int value) {
            this.writeMemberStart(name);
            JsonBuilder.this.jw.writeNumber(value);
        }

        @Override
        public void writeNumberMember(String name, long value) {
            this.writeMemberStart(name);
            JsonBuilder.this.jw.writeNumber(value);
        }

        @Override
        public void writeNumberMember(String name, float value) {
            this.writeMemberStart(name);
            JsonBuilder.this.jw.writeNumber(value);
        }

        @Override
        public void writeNumberMember(String name, double value) {
            this.writeMemberStart(name);
            JsonBuilder.this.jw.writeNumber(value);
        }

        @Override
        public void writeBooleanMember(String name, boolean value) {
            this.writeMemberStart(name);
            JsonBuilder.this.jw.writeBoolean(value);
        }

        @Override
        public void writeNullMember(String name) {
            this.writeMemberStart(name);
            JsonBuilder.this.jw.writeNull();
        }
    }

    private class ObjectWriter extends AbstractJsonWriter {
        private final ObjectOrArrayWriter parent;

        public ObjectWriter(ObjectOrArrayWriter parent) {
            this.parent = parent;
        }

        @Override
        public void writeMemberStart(String name) {
            JsonBuilder.this.jw.writeFieldName(name);
        }

        @Override
        public void writeStart() {
            JsonBuilder.this.jw.writeObjectStart();
        }

        @Override
        public void writeEnd() {
            JsonBuilder.this.jw.writeObjectEnd();
        }

        @Override
        public ObjectOrArrayWriter getParent() {
            return parent;
        }
    }

    private class ArrayWriter extends AbstractJsonWriter {
        private final ObjectOrArrayWriter parent;

        public ArrayWriter(ObjectOrArrayWriter parent) {
            this.parent = parent;
        }

        @Override
        public void writeMemberStart(String name) {
        }

        @Override
        public void writeStart() {
            JsonBuilder.this.jw.writeArrayStart();
        }

        @Override
        public void writeEnd() {
            JsonBuilder.this.jw.writeArrayEnd();
        }

        @Override
        public ObjectOrArrayWriter getParent() {
            return parent;
        }
    }

    private class RootWriter extends AbstractJsonWriter {
        @Override
        public void writeMemberStart(String name) {
        }

        @Override
        public void writeStart() { throw new UnsupportedOperationException(); }

        @Override
        public void writeEnd() { throw new UnsupportedOperationException(); }

        @Override
        public ObjectOrArrayWriter getParent() { throw new UnsupportedOperationException(); }
    }

}
