package ru.yandex.calendar.ammo

import ru.yandex.misc.cmdline.CmdArgsChief
import ru.yandex.misc.cmdline.CmdLineArgs
import ru.yandex.misc.cmdline.Parameter
import kotlin.system.exitProcess

fun main(arguments: Array<String>) {
    val args = CmdArgsChief.parse(Args::class.java, arguments)

    val parser = when (args.inputFormat) {
        Args.InputFormat.NGINX -> NginxAccessLogParser()
        Args.InputFormat.CALENDAR -> CalendarRequestsLogParser()
    }
    val processor = when (args.processor) {
        Args.Processor.WEB_RO -> WebReadOnlyProcessor()
        Args.Processor.WEB_OMIT_ICS -> WebOmitIcsUrlProcessor()
    }
    val serializer = when (args.outputFormat) {
        Args.OutputFormat.URI_GET -> UriGetAmmoSerializer()
        Args.OutputFormat.HTTP_JSON -> HttpJsonAmmoSerializer()
    }

    for (line in generateSequence(::readLine)) {
        val parsed = parser.parse(line) ?: continue
        val processed = processor.process(parsed) ?: continue
        val serialized = serializer.serialize(processed)

        println(serialized)

        if (System.out.checkError()) {
            exitProcess(-1)
        }
    }
}

@CmdLineArgs
class Args {

    @field:Parameter(names = ["-f", "--from"], description = "format of input")
    val inputFormat = InputFormat.NGINX

    @field:Parameter(names = ["-t", "--to"], description = "format of output")
    val outputFormat = OutputFormat.URI_GET

    @field:Parameter(names = ["-p", "--processor"], description = "processor type")
    val processor = Processor.WEB_RO

    enum class InputFormat {
        NGINX,
        CALENDAR,
    }

    enum class OutputFormat {
        URI_GET,
        HTTP_JSON,
    }

    enum class Processor {
        WEB_RO,
        WEB_OMIT_ICS,
    }
}
