package ru.yandex.calendar.ammo

import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.node.ObjectNode
import org.apache.http.message.BasicLineParser
import java.net.URI

data class ParsedRequest(
    val uri: URI,
    val body: String,
)

abstract class BaseJsonLogParser {

    private val parser = ObjectMapper().readerFor(ObjectNode::class.java)

    fun parse(line: String): ParsedRequest? {
        if (line.isEmpty()) return null

        return parse(parser.readValue<ObjectNode>(line))
    }

    protected abstract fun parse(node: ObjectNode): ParsedRequest?

    @Suppress("SameParameterValue")
    protected fun requireField(node: ObjectNode, field: String): JsonNode {
        return requireNotNull(node.get(field)) { "Missing $field" }
    }

    protected fun requireTextField(node: ObjectNode, field: String): String {
        return requireNotNull(node.get(field)?.textValue()) { "Missing $field" }
    }
}

/**
 * Export in json line format content of
 * https://yt.yandex-team.ru/hahn/navigation?path=//logs/calendar-public-nginx-access-log
 */
class NginxAccessLogParser : BaseJsonLogParser() {

    override fun parse(node: ObjectNode): ParsedRequest {
        val request = requireTextField(node, "request")
        val line = BasicLineParser.parseRequestLine(request, BasicLineParser())

        val uri = URI(line.uri)
        val body = requireTextField(node, "request_body")

        return ParsedRequest(uri, body)
    }
}

/**
 * @see ru.yandex.calendar.logic.log.requests.RequestsLogger
 *
 * Unlike nginx format has ability to construct ByIcsUrl requests
 */
class CalendarRequestsLogParser : BaseJsonLogParser() {

    override fun parse(node: ObjectNode): ParsedRequest {

        val uri = URI(requireTextField(node, "uri"))
        val body = requireField(node, "body").textValue() ?: ""

        return ParsedRequest(uri, body)
    }
}
