package ru.yandex.calendar.ammo

import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.node.JsonNodeFactory
import com.fasterxml.jackson.databind.node.ObjectNode
import com.fasterxml.jackson.databind.node.TextNode
import java.lang.IllegalStateException

/**
 * @url https://wiki.yandex-team.ru/load/guides/ammo/
 * @url https://wiki.yandex-team.ru/load/pandora/#formatpatronov
 */
interface AmmoSerializer {

    fun serialize(request: ProcessedRequest): String
}

abstract class BaseUriAmmoSerializer : AmmoSerializer {

    private val headers = hashMapOf<String, String>()

    protected abstract fun serializePayload(request: ProcessedRequest): String

    final override fun serialize(request: ProcessedRequest): String {
        val result = StringBuilder()

        for ((name, value) in request.headers) {
            if (headers[name] != value) {
                headers[name] = value
                result.append("[$name: $value]\n")
            }
        }
        result.append(serializePayload(request))

        return result.toString()
    }
}

class UriGetAmmoSerializer : BaseUriAmmoSerializer() {

    override fun serializePayload(request: ProcessedRequest): String {
        if (request.method != "GET") {
            throw IllegalStateException(
                "Attempt to serialize ${request.method} ${request.uri.rawPath} request as uri-get"
            )
        }
        return if (request.tag != null) {
            "${request.uri.toASCIIString()} ${request.tag}"
        } else {
            request.uri.toASCIIString()
        }
    }
}

class HttpJsonAmmoSerializer : AmmoSerializer {

    override fun serialize(request: ProcessedRequest): String {
        val node = ObjectNode(JsonNodeFactory.instance)

        node.put("method", request.method)
        node.put("uri", request.uri.toASCIIString())

        if (request.tag != null) {
            node.put("tag", request.tag)
        }
        if (request.headers.isNotEmpty()) {
            node.replace("headers", ObjectNode(
                JsonNodeFactory.instance,
                request.headers.mapValues { TextNode.valueOf(it.value) },
            ))
        }
        if (request.body != null) {
            node.put("body", request.body)
        }
        return node.toString()
    }
}
