package main

import (
	"bufio"
	"database/sql"
	"fmt"
	"log"
	"net/http"
	"os"
	"strconv"
	"sync"

	"github.com/jackc/pgx"
)

var wg sync.WaitGroup
var lock = sync.Mutex{}
var logEmails = []string{}

//nolint:ST1003
var byUid = map[int64]User{}
var byEmail = map[string]User{}

type User struct {
	//nolint:ST1003
	Uid         int64
	Email       sql.NullString
	YandexEmail sql.NullString
	Domain      sql.NullString
	UserLogin   sql.NullString
	UserName    sql.NullString
	Language    sql.NullString
}

type Args struct {
	//nolint:ST1003
	Uid      string
	Login    string
	Emails   string
	Dbfields string
}

func UserinfoHandler(w http.ResponseWriter, r *http.Request) {
	args := &Args{
		Uid:      r.FormValue("uid"),
		Login:    r.FormValue("login"),
		Emails:   r.FormValue("emails"),
		Dbfields: r.FormValue("dbfields"),
	}
	w.Header().Set("Content-Type", "text/xml")

	if string(args.Emails) == "getall" {
		UserinfoGetall(w, args)
	} else {
		Userinfo(w, args)
	}
}

func Userinfo(w http.ResponseWriter, args *Args) {
	var err error
	if val, ok := byEmail[string(args.Login)]; ok {
		_, err = fmt.Fprintf(w, `<?xml version="1.0" encoding="UTF-8"?>
<doc>
<uid hosted="1" domid="111111" domain="`+val.Domain.String+`" mx="1" domain_ena="1" catch_all="0">%d</uid>
<login>%s</login>
<have_password>1</have_password>
<have_hint>0</have_hint>
<karma confirmed="0">0</karma>
<karma_status>0</karma_status>
<dbfield id="accounts.login.uid">%s</dbfield>
</doc>%s`, val.Uid, val.UserLogin.String, val.UserLogin.String, "\n")
	} else {
		_, err = fmt.Fprintf(w, `<?xml version="1.0" encoding="UTF-8"?>
<doc>
<uid hosted="0"></uid>
<karma confirmed="0">0</karma>
<karma_status>0</karma_status>
</doc>%s`, "\n")
	}
	if err != nil {
		log.Println(err)
	}
}

func LoginHandler(w http.ResponseWriter, r *http.Request) {
	login := string(r.FormValue("login"))
	w.Header().Set("Content-Type", "text/xml")

	var err error
	if val, ok := byEmail[login]; ok {

		_, err = fmt.Fprintf(w, `<?xml version="1.0" encoding="UTF-8"?>
<doc>
<status id="0">VALID</status>
<error>OK</error>
<uid hosted="1" domid="111111" domain="`+val.Domain.String+`" mx="1" domain_ena="1" catch_all="0">%d</uid>
<login>%s</login>
<have_password>1</have_password>
<have_hint>0</have_hint>
<karma confirmed="0">0</karma>
<karma_status>0</karma_status>
</doc>%s`, val.Uid, val.UserLogin.String, "\n")
	} else {
		_, err = fmt.Fprintf(w, `<?xml version="1.0" encoding="UTF-8"?>
<doc>
<status id="2">INVALID</status>
<error>Login not found</error>
</doc>%s`, "\n")
	}
	if err != nil {
		log.Println(err)
	}
}

func UserinfoGetall(w http.ResponseWriter, args *Args) {
	uid, _ := strconv.ParseInt(string(args.Uid), 10, 64)

	var err error
	if val, ok := findOrGenerateUser(uid); ok {
		_, err = fmt.Fprintf(w, `<?xml version="1.0" encoding="UTF-8"?>
<doc>
<uid hosted="1" domid="111111" domain="`+val.Domain.String+`" mx="1" domain_ena="1" catch_all="0">%d</uid>
<login>`+val.UserLogin.String+`</login>
<have_password>1</have_password>
<have_hint>1</have_hint>
<karma confirmed="0">0</karma>
<karma_status>6000</karma_status>
<dbfield id="account_info.fio.uid">`+val.UserName.String+`</dbfield>
<dbfield id="accounts.login.uid">`+val.UserLogin.String+`</dbfield>
<dbfield id="userinfo.lang.uid">`+val.Language.String+`</dbfield>
<address-list>
<address validated="1" default="1" rpop="0" silent="0" unsafe="0" native="1" born-date="2013-06-04 09:01:33">`+val.Email.String+`</address>
</address-list>
</doc>%s`, val.Uid, "\n")
	} else {
		_, err = fmt.Fprintf(w, `<?xml version="1.0" encoding="UTF-8"?>
<doc>
<uid hosted="0"></uid>
<karma confirmed="0">0</karma>
<karma_status>0</karma_status>
</doc>%s`, "\n")
	}
	if err != nil {
		log.Println(err)
	}
}

func findOrGenerateUser(uid int64) (User, bool) {
	if user, ok := byUid[uid]; ok {
		return user, true
	} else {
		return generateUser(uid), true
	}
}

func generateUser(uid int64) User {
	login := "yandex-load-" + strconv.FormatInt(uid, 10)

	var domain string
	if uid >= 1130000000000000 {
		domain = "somewhere.com"
	} else if uid >= 1120000000000000 {
		domain = "yandex-team.ru"
	} else if uid >= 1110000000000000 {
		domain = "kinopoisk.ru"
	} else {
		domain = "yandex.ru"
	}

	nullStr := func(value string) sql.NullString {
		return sql.NullString{String: value, Valid: true}
	}

	return User{
		Uid:         uid,
		Email:       nullStr(login + "@" + domain),
		YandexEmail: nullStr(login + "@" + domain),
		Domain:      nullStr(domain),
		UserLogin:   nullStr(login),
		UserName:    nullStr(login),
		Language:    nullStr("ru"),
	}
}

func parseBatch(rows *pgx.Rows) {
	user := User{}
	for rows.Next() {
		err := rows.Scan(&user.Uid, &user.Email, &user.YandexEmail, &user.Domain, &user.UserLogin, &user.UserName, &user.Language)
		if err != nil {
			log.Println(err)
		}
		lock.Lock()
		byUid[user.Uid] = user
		byEmail[user.Email.String] = user
		lock.Unlock()
	}
}

func InitFromDB() {
	conn := MakePgConnection()
	var tableLength int
	var result *pgx.Rows
	batch := 0
	fmt.Println("got_connection")
	// select a row
	err := conn.QueryRow("select count(uid) from settings").Scan(&tableLength)
	if err != nil {
		log.Fatal(err)
	}
	//var user User
	for batch <= tableLength {
		wg.Add(10)
		for i := 0; i < 10; i++ {
			go func(batch int, wg *sync.WaitGroup) {
				defer wg.Done()
				result, err = conn.Query("select uid, email, YandexEmail, domain, UserLogin, UserName, language from settings offset $1 limit 1000", batch)
				if err != nil {
					log.Fatal(err)
				}
				parseBatch(result)
			}(batch, &wg)
			batch += 1000
		}
		wg.Wait()
		fmt.Println(batch, " - ", tableLength)
	}
	fmt.Println("initialized from DB", len(byUid), tableLength)
}
func parseEmails() {
	file, err := os.Open("/etc/yamail/emails.csv")
	if err != nil {
		fmt.Println(err)
	}

	defer file.Close()

	scanner := bufio.NewScanner(file)
	for scanner.Scan() {
		logEmails = append(logEmails, scanner.Text())
	}

}

func InitByCSVfromDB() {
	parseEmails()
	conn := MakePgConnection()
	var tableLength int
	fmt.Println("got_connection")
	// select a row
	err := conn.QueryRow("select count(uid) from settings").Scan(&tableLength)
	if err != nil {
		log.Fatal(err)
	}
	//var user User
	for _, mail := range logEmails {
		user := User{}
		err = conn.QueryRow("select uid, email, yandex_email, domain, user_login, user_name, language from settings where email = $1", mail).Scan(&user.Uid, &user.Email, &user.YandexEmail, &user.Domain, &user.UserLogin, &user.UserName, &user.Language)
		if err != nil {
			//fmt.Println("query error", mail, len(logEmails))
		} else {
			lock.Lock()
			//fmt.Println("good, ", user.Email.String)
			byUid[user.Uid] = user
			byEmail[user.Email.String] = user
			lock.Unlock()
		}
	}
	fmt.Println("initialized from Csv DB", len(byUid), len(logEmails))
}

func main() {
	InitByCSVfromDB()
	RunServer(":8080")
}
