package ru.yandex.calendar.micro.perm;

/**
 * Event atomic actions / permissions
 */
public enum EventAction {
    /**
     * <b>Owners</b>: event creator, any user (E.perm_all == 'view'), any participant or
     * owner of {@link #EDIT}.<br/>
     *
     * <i>In fact, a person that has an invitation to a meeting, also
     * owns permission to this action, but this is needed only when
     * working with invitations, so according to performance reasons,
     * we do not add a {@link #VIEW} check there.</i><br/>
     *
     * <b>Required to</b>: view all event details, including invited people and participant lists.<br/>
     * <b>Checked when</b>: viewing single event / events at grid / external calendar blocks;
     * also when attempting to attach this event to some layer.
     */
    VIEW,
    /**
     * <b>Owners</b>: event creator, participant (E.perm_participants == 'edit') or
     * owner of {@link LayerAction#EDIT_EVENT} evaluated on event container layer.<br/>
     * <b>Required to</b>: edit an event (besides permissions, for which
     * {@link #EDIT_PERMISSIONS} should be used instead).<br/>
     * <b>Checked when</b>: user edits any instance: own or other user's,
     * primary or secondary (attached).<br/>
     *
     * <i>When editing an event, a layer selection for this instance is available.
     * It is important to understand layer change rules for event instances. For now:
     * </i>
     * <ul>
     * <li> for primary or secondary (attached) instance laying at my layer,
     * allow to move this instance between my layers (it is safe from
     * the permissions and instance collisions points of view. Otherwise,
     * we would need to check delete/create or detach/attach permissions,
     * as well as check if event instance already exists at the new layer);
     * </li>
     * <li> for primary and secondary (attached) instances laying at somebody
     * else's layer, it is unsafe to move them even between that user's
     * layers, because, besides instance collisions, permissions failure
     * can occur (complex checks might be unacceptable to make at makeup).
     * </li>
     * </ul>
     * <i>The rules above are quite easy to check at the makeup, and provide acceptable
     * usability.
     * </i>
     */
    EDIT,
    /**
     * <b>Owners</b>: event creator, owner of {@link LayerAction#EDIT_EVENT} evaluated on event
     * container layer. Unlike {@link #EDIT}, participants never can edit permissions.<br/>
     * <b>Required to</b>: edit event permissions (for all, for participants, p.-can-invite) and
     * remove guests.<br/>
     * <b>Checked when</b>: user edits any instance: own or other user's,
     * primary or secondary (attached).<br/>
     */
    EDIT_PERMISSIONS,
    /**
     * <b>Owners</b>: event creator or owner of {@link LayerAction#DELETE_EVENT}
     * evaluated on event container layer.<br/>
     * <b>Required to</b>: delete an event, including its primary and all secondary instances.<br/>
     * <b>Checked when</b>: user deletes a primary event instance.<br/>
     *
     * <i>For details, see {@link LayerAction#DELETE_EVENT}.</i>
     */
    DELETE,
    /**
     * <b>Owners</b>: anybody, depends on event itself.<br/>
     * <b>Required to</b>: split event when doing other actions
     * (for which separate permissions are needed!)<br/>
     * <b>Checked when</b>: event is edited or deleted.<br/>
     *
     * <i>Note that this permission is just an <b>addition</b> to others,
     * like {@link #EDIT} or {@link #DELETE}.</i>
     */
    SPLIT,
    /**
     * <b>Owners</b>: event creator, participant (E.participants_invite == true).<br/>
     *
     * <i>Note that additional condition (EL.invitation_level <= 1) may be added later.
     * Invitation level is a degree of separation between event creator and this
     * participant user (0 for creator user, 1 for guests invited by creator, etc.)</i><br/>
     *
     * <b>Required to</b>: invite a new user (email) to participate at this meeting.<br/>
     * <b>Checked when</b>: user creates a new invitation for some new user (email).<br/>
     */
    INVITE,
    /**
     * <b>Owners</b>: event creator, any participant (user who has EventLayer
     * instance in his layers, marked with 'goes_to_meeting').<br/>
     * <b>Required to</b>: determine whether user participates in this event<br/>
     * <b>Checked when</b>: <i>calculating other event permissions</i>
     */
    PARTICIPATE,
    CHANGE_ORGANIZER;

    public String toXmlName() {
        switch (this) {
            case VIEW:             return "view";
            case EDIT:             return "edit";
            case EDIT_PERMISSIONS: return "edit-permissions";
            case DELETE:           return "delete";
            case SPLIT:            return "split";
            case INVITE:           return "invite";
            case PARTICIPATE:      return "participate";
            case CHANGE_ORGANIZER: return "change-organizer";
        }

        throw new IllegalStateException("Unexpected event action");
    }
}
