package ru.yandex.calendar.micro.perm;

/**
 * Layer atomic actions / permissions.
 * There is no GRANT_PERMISSIONS, because we do not have re-sharing
 * (only creator can grant and revoke permissions).
 */
public enum LayerAction {
    /**
     * <b>Owners</b>: layer creator, any user (L.perm_all == 'list'), any layer owner.<br/>
     * <b>Required to</b>: have access to layer to view its own data, as well as events
     * that are allowed to see.<br/>
     * <b>Checked when</b>: user gets layer contents (grid, ext. methods).
     */
    LIST,
    /**
     * Same as LIST, but also signals about anonymous 'self-sharing'
     */
    ANONYMOUS_LIST,
    /**
     * <b>Owners</b>: layer creator, layer owner (LU.perm in 'admin', 'edit', 'create').<br/>
     * <b>Required to</b>: create a new event at this layer.<br/>
     * <b>Checked when</b>: user creates new event (maybe, also before that).
     */
    CREATE_EVENT,
    /**
     * <b>Owners</b>: layer creator, layer owner (LU.perm in 'admin', 'edit', 'create', 'view').<br/>
     * <b>Required to</b>: view any event whose primary instance lays at this layer.<br/>
     * <b>Checked when</b>: attempting to view event instance (which may lay at other layer).<br/>
     */
    VIEW_EVENT,
    /**
     * <b>Owners</b>: layer creator, layer owner (LU.perm in 'admin', 'edit').<br/>
     * <b>Required to</b>: edit any event whose primary instance lays at this layer.<br/>
     * <b>Checked when</b>: attempting to edit event instance (which may lay at other layer).<br/>
     *
     * <i>For event editing details, see {@link EventAction#EDIT}.</i>
     */
    EDIT_EVENT,
    /**
     * <b>Owners</b>: layer creator, layer owner (LU.perm in 'admin', 'edit').<br/>
     * <b>Required to</b>: detach a secondary event instance from this layer.<br/>
     * <b>Checked when</b>: detaching event (including a meeting detaching case).
     */
    DETACH_EVENT,
    /**
     * <b>Owners</b>: layer creator, layer owner (LU.perm in 'admin', 'edit').<br/>
     * <b>Required to</b>: delete any event whose primary instance lays at this layer.<br/>
     * <b>Checked when</b>: attempting to delete event instance.<br/>
     *
     * <i>Note: the UI decision about situation when both detach and delete of the
     * secondary event instance is available to the user, we always select detach.
     * Therefore, in the case of the deleting event, its (primary) instance must
     * lay at this (container) layer (TODO is that still true as of July 2010?)</i>
     */
    DELETE_EVENT,
    /**
     * <b>Owners</b>: layer creator, layer owner (LU.perm in 'edit', 'admin').<br/>
     * <b>Required to</b>: change common layer details, manipulating events.<br/>
     * <b>Checked when</b>: user edits common layer details ('name' field - NOTE:
     * not 'L.perm_all' field), or edits events (NOTE: no check for event-users).
     */
    EDIT,
    /**
     * <b>Owners</b>: layer creator, layer owner (LU.perm in 'admin').<br/>
     * <b>Required to</b>: delete layer.<br/>
     * <b>Checked when</b>: user deletes layer.
     */
    DELETE,
    /**
     * <b>Owners</b>: layer creator. NOW: and also layer owner (LU.perm in 'admin').<br/>
     * <b>Required to</b>: give access (of any power) to this layer to somebody else.<br/>
     * <b>Checked when</b>: obtaining private token, giving layer permissions to user,
     * changing L.perm_all field
     */
    GRANT;

    public String toXmlName() {
        switch (this) {
            case LIST:         return "list";
            case CREATE_EVENT: return "create-event";
            case VIEW_EVENT:   return "view-event";
            case EDIT_EVENT:   return "edit-event";
            case DETACH_EVENT: return "detach-event";
            case DELETE_EVENT: return "delete-event";
            case EDIT:         return "edit";
            case DELETE:       return "delete";
            case GRANT:        return "grant";
        }

        throw new IllegalStateException("Unexpected layer action");
    }
}
