package ru.yandex.calendar.micro.yt;

import io.micronaut.context.annotation.Requires;
import io.micronaut.context.annotation.Value;
import lombok.AccessLevel;
import lombok.experimental.FieldDefaults;
import reactor.core.publisher.Flux;
import ru.yandex.calendar.micro.yt.entity.YtDepartment;
import ru.yandex.calendar.micro.yt.exception.InconsistentYtDepartmentException;
import ru.yandex.mail.cerberus.client.GroupClient;
import ru.yandex.mail.cerberus.client.dto.Group;
import ru.yandex.mail.cerberus.yt.data.YtDepartmentInfo;
import ru.yandex.mail.micronaut.common.JsonMapper;
import ru.yandex.mail.micronaut.common.RawJsonString;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.Duration;
import java.util.List;

import static ru.yandex.mail.cerberus.yt.staff.StaffConstants.YT_DEPARTMENT_GROUP_TYPE;

@Singleton
@Requires(env = "yt")
@FieldDefaults(makeFinal = true, level = AccessLevel.PRIVATE)
public class YtDepartmentClient {
    GroupClient client;
    JsonMapper jsonMapper;
    YtCerberusFetcher<Group<RawJsonString>, YtDepartment> fetcher;

    @Inject
    public YtDepartmentClient(GroupClient client, JsonMapper jsonMapper, Throttle throttle,
                              @Value("${calendar.yt.department-page-size}") int pageSize) {
        this.client = client;
        this.jsonMapper = jsonMapper;
        this.fetcher = new YtCerberusFetcher<>(throttle, pageSize, this::map, this::fetch);
    }

    private YtDepartment map(Group<RawJsonString> group) {
        final var info = group.getInfo()
            .map(node -> jsonMapper.fromJson(node, YtDepartmentInfo.class))
            .orElseThrow(InconsistentYtDepartmentException::new);
        return new YtDepartment(group.getId(), group.getName(), group.isActive(), info);
    }

    private Flux<List<Group<RawJsonString>>> fetch(int pageSize) {
        return client.findGroupsByTypeRx(YT_DEPARTMENT_GROUP_TYPE, pageSize);
    }

    public Flux<List<YtDepartment>> allDepartmentsBatches(Duration delay) {
        return fetcher.fetchAllBatches(delay);
    }

    public Flux<YtDepartment> allDepartments(Duration delay) {
        return fetcher.fetchAll(delay);
    }
}
