package ru.yandex.calendar.micro.yt;

import io.micronaut.context.annotation.Requires;
import lombok.AccessLevel;
import lombok.experimental.FieldDefaults;
import reactor.core.publisher.Flux;
import ru.yandex.calendar.micro.yt.entity.YtOffice;
import ru.yandex.mail.micronaut.common.JsonMapper;
import ru.yandex.mail.cerberus.client.LocationClient;
import ru.yandex.mail.cerberus.client.dto.Location;
import ru.yandex.mail.cerberus.yt.data.YtOfficeInfo;
import ru.yandex.mail.micronaut.common.RawJsonString;
import ru.yandex.calendar.micro.yt.exception.InconsistentYtOfficeException;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.Duration;
import java.util.List;

import static ru.yandex.mail.cerberus.yt.staff.StaffConstants.YT_OFFICE_TYPE;

@Singleton
@Requires(env = "yt")
@FieldDefaults(makeFinal = true, level = AccessLevel.PRIVATE)
public class YtLocationClient {
    JsonMapper mapper;
    LocationClient client;
    YtCerberusFetcher<Location<RawJsonString>, YtOffice> fetcher;

    @Inject
    public YtLocationClient(JsonMapper mapper, LocationClient client, Throttle throttle) {
        this.mapper = mapper;
        this.client = client;
        this.fetcher = new YtCerberusFetcher<>(throttle, Integer.MAX_VALUE, this::map, this::fetch);
    }

    private YtOffice map(Location<RawJsonString> location) {
        final var info = location.getInfo().map(node -> mapper.fromJson(node, YtOfficeInfo.class))
            .orElseThrow(InconsistentYtOfficeException::new);
        return new YtOffice(location.getId(), location.getName(), info);
    }

    private Flux<List<Location<RawJsonString>>> fetch(int pageSize) {
        return client.findLocationsByTypeRx(YT_OFFICE_TYPE, Integer.MAX_VALUE);
    }

    public Flux<List<YtOffice>> allOfficesBatches(Duration delay) {
        return fetcher.fetchAllBatches(delay);
    }

    public Flux<YtOffice> allOffices(Duration delay) {
        return fetcher.fetchAll(delay);
    }
}
