package ru.yandex.calendar.micro.yt;

import io.micronaut.context.annotation.Requires;
import io.micronaut.context.annotation.Value;
import lombok.AccessLevel;
import lombok.experimental.FieldDefaults;
import reactor.core.publisher.Flux;
import ru.yandex.calendar.micro.yt.entity.YtRoom;
import ru.yandex.calendar.micro.yt.exception.InconsistentYtRoomException;
import ru.yandex.mail.cerberus.LocationKey;
import ru.yandex.mail.micronaut.common.JsonMapper;
import ru.yandex.mail.cerberus.client.ResourceClient;
import ru.yandex.mail.cerberus.client.dto.Resource;
import ru.yandex.mail.cerberus.yt.data.YtRoomInfo;
import ru.yandex.mail.micronaut.common.RawJsonString;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.Duration;
import java.util.List;

import static ru.yandex.mail.cerberus.yt.staff.StaffConstants.YT_ROOM_RESOURCE_TYPE_NAME;

@Singleton
@Requires(env = "yt")
@FieldDefaults(makeFinal = true, level = AccessLevel.PRIVATE)
public class YtResourceClient {
    JsonMapper mapper;
    ResourceClient client;
    YtCerberusFetcher<Resource<RawJsonString>, YtRoom> fetcher;

    @Inject
    public YtResourceClient(JsonMapper mapper, ResourceClient client, Throttle throttle,
                            @Value("${calendar.yt.room-page-size}") int pageSize) {
        this.mapper = mapper;
        this.client = client;
        this.fetcher = new YtCerberusFetcher<>(throttle, pageSize, this::map, this::fetch);
    }

    private YtRoom map(Resource<RawJsonString> resource) {
        final var info = resource.getInfo().map(node -> mapper.fromJson(node, YtRoomInfo.class))
                .orElseThrow(InconsistentYtRoomException::new);
        return new YtRoom(resource.getId(), resource.getName(), resource.getLocation().map(LocationKey::getId), resource.isActive(), info);
    }

    private Flux<List<Resource<RawJsonString>>> fetch(int pageSize) {
        return client.findResourcesByTypeRx(YT_ROOM_RESOURCE_TYPE_NAME, pageSize);
    }

    public Flux<List<YtRoom>> allResourcesBatches(Duration delay) {
        return fetcher.fetchAllBatches(delay);
    }

    public Flux<YtRoom> allResources(Duration delay) {
        return fetcher.fetchAll(delay);
    }
}
