package ru.yandex.calendar.micro.yt;

import io.micronaut.context.annotation.Requires;
import io.micronaut.context.annotation.Value;
import lombok.AccessLevel;
import lombok.experimental.FieldDefaults;
import lombok.val;
import one.util.streamex.StreamEx;
import reactor.core.publisher.Flux;
import ru.yandex.calendar.micro.yt.entity.YtUser;
import ru.yandex.calendar.micro.yt.entity.YtUserWithDepartmentIds;
import ru.yandex.calendar.micro.yt.exception.InconsistentYtUserException;
import ru.yandex.mail.cerberus.GroupKey;
import ru.yandex.mail.cerberus.client.dto.UserWithGroups;
import ru.yandex.mail.micronaut.common.JsonMapper;
import ru.yandex.mail.cerberus.UserType;
import ru.yandex.mail.cerberus.client.UserClient;
import ru.yandex.mail.cerberus.client.dto.User;
import ru.yandex.mail.cerberus.yt.data.YtUserInfo;
import ru.yandex.mail.micronaut.common.RawJsonString;

import javax.inject.Inject;
import javax.inject.Singleton;
import java.time.Duration;
import java.util.List;

import static ru.yandex.mail.cerberus.yt.staff.StaffConstants.YT_DEPARTMENT_GROUP_TYPE;

@Singleton
@Requires(env = "yt")
@FieldDefaults(makeFinal = true, level = AccessLevel.PRIVATE)
public class YtUserClient {
    UserClient client;
    JsonMapper mapper;
    YtCerberusFetcher<UserWithGroups<RawJsonString>, YtUserWithDepartmentIds> fetcher;

    @Inject
    public YtUserClient(UserClient client, JsonMapper mapper, Throttle throttle,
                        @Value("${calendar.yt.user-page-size}") int pageSize) {
        this.client = client;
        this.mapper = mapper;
        this.fetcher = new YtCerberusFetcher<>(throttle, pageSize, this::map, this::fetch);
    }

    private YtUser map(User<RawJsonString> user) {
        final var info = user.getInfo().map(node -> mapper.fromJson(node, YtUserInfo.class))
            .orElseThrow(InconsistentYtUserException::new);
        return new YtUser(user.getUid(), user.getLogin(), info);
    }

    private YtUserWithDepartmentIds map(UserWithGroups<RawJsonString> userWithGroups) {
        val user = map(userWithGroups.getUser());
        val departmentIds = StreamEx.of(userWithGroups.getGroups())
            .filterBy(GroupKey::getType, YT_DEPARTMENT_GROUP_TYPE)
            .map(GroupKey::getId)
            .toImmutableSet();
        return new YtUserWithDepartmentIds(user, departmentIds);
    }

    private Flux<List<UserWithGroups<RawJsonString>>> fetch(int pageSize) {
        return client.usersWithGroupsRx(UserType.YT, pageSize);
    }

    public Flux<List<YtUserWithDepartmentIds>> allUsersWithDepartmentIdsBatches(Duration delay) {
        return fetcher.fetchAllBatches(delay);
    }

    public Flux<YtUserWithDepartmentIds> allUsersWithDepartmentIds(Duration delay) {
        return fetcher.fetchAll(delay);
    }
}
