if cjson == nil then
    cjson = require "cjson"
end


local HISTOGRAM_LOG_BASE = 1.5
local HISTOGRAM_MIN_LOG = -50
local HISTOGRAM_MAX_LOG = 50
local HISTOGRAM_BORDERS = {0}
for idx = -50, 50 do
    HISTOGRAM_BORDERS[idx+50+2] = 1.5 ^ idx
end

local HISTOGRAM_VALUES = {}
local HISTOGRAM_NAMES = {}
local NUMERIC_METRICS = {}
local TRANSFER_DELAY = 1.0


local function increment_dict_value(dict, key, value)
    if dict[key] == nil then
        dict[key] = value
    else
        dict[key] = dict[key] + value
    end
end


function increment_metric(name, value)
    if value == nil then
        ngx.log(ngx.ERR, "Trying to set nil value for metric " .. name)
    else
        increment_dict_value(NUMERIC_METRICS, name, value)
    end
end


function add_to_histogram(name, value)
    local offset = 0
    if value == nil then
        ngx.log(ngx.ERR, "Trying to add nil value to histogram for metric " .. name)
        return
    elseif value ~= 0 then
        offset = math.floor(math.max(HISTOGRAM_MIN_LOG - 1, math.min(HISTOGRAM_MAX_LOG, math.log(value) / math.log(HISTOGRAM_LOG_BASE))) - HISTOGRAM_MIN_LOG + 1) + 1
    end

    increment_dict_value(HISTOGRAM_VALUES, name .. offset, 1)
    HISTOGRAM_NAMES[name] = 1
end


local function yasmify_histogram(name)
    local hvalues = HISTOGRAM_VALUES
    local output = {}
    for idx = 1, #HISTOGRAM_BORDERS do
        local value = HISTOGRAM_VALUES[name .. idx]
        if value ~= nil then
            output[#output+1] = {HISTOGRAM_BORDERS[idx], value}
        end
    end
    return output
end

function get_yasm_metrics()
    local output = {}
    for key, val in pairs(NUMERIC_METRICS) do
        output[#output + 1] = { key, val }
    end

    for key, val in pairs(HISTOGRAM_NAMES) do
        local value = yasmify_histogram(key)
        output[#output + 1] = { key, value }
    end
    return output
end

function update_web_metrics()
    increment_metric(string.format("nginx_request_web_%dxx_count_deee", math.floor(ngx.status / 100)), 1)    
    increment_metric(string.format("nginx_request_web_%s_%dxx_count_deee", ngx.var.uri, math.floor(ngx.status / 100)), 1)
    increment_metric(string.format("nginx_request_web_%s_%d_count_deee", ngx.var.uri, ngx.status), 1)
    add_to_histogram("nginx_request_time_web_hgram", tonumber(ngx.var.request_time))
    add_to_histogram(string.format("nginx_request_time_web_%s_hgram", ngx.var.uri), tonumber(ngx.var.request_time))
end

function update_caldav_metrics()
    increment_metric(string.format("nginx_request_caldav_%dxx_count_deee", math.floor(ngx.status / 100)), 1)
    add_to_histogram("nginx_request_time_caldav_hgram", tonumber(ngx.var.request_time))
    local strip_uri = string.sub(ngx.var.uri, 2, string.len(ngx.var.uri))
    local end_pos = strip_uri:find("/")
    local route = strip_uri
    if end_pos ~= nil then
        route = string.sub(strip_uri, 1, end_pos - 1)
    end

    increment_metric(string.format("nginx_request_caldav_%s_%s_%dxx_count_deee", ngx.var.request_method:lower(), route, math.floor(ngx.status / 100)), 1)
    increment_metric(string.format("nginx_request_caldav_%s_%s_%d_count_deee", ngx.var.request_method:lower(), route, ngx.status), 1)
    add_to_histogram(string.format("nginx_request_time_caldav_%s_hgram", route), tonumber(ngx.var.request_time)) 
end
