import json
import requests

from ticket_parser2.api.v1 import BlackboxClientId, TvmClient


def get_qloud_ip():
    with open('/etc/qloud/meta.json') as fd:
        return json.load(fd)['ipv6'].split('/')[0]


class BlackboxClient:
    BLACKBOX_URLS = {
        BlackboxClientId.Prod: 'https://blackbox.yandex.net/blackbox',
        BlackboxClientId.ProdYateam: 'https://blackbox.yandex-team.ru/blackbox',
        BlackboxClientId.Test: 'https://blackbox-test.yandex.net/blackbox',
        BlackboxClientId.TestYateam: 'https://blackbox-test.yandex-team.ru/blackbox',
    }

    def __init__(self, tvm_client: TvmClient, env: BlackboxClientId, url: str = None):
        self.tvm_client = tvm_client
        self.env = env
        if url:
            self.url = url
        else:
            self.url = self.BLACKBOX_URLS[self.env]

    def oauth(self, oauth_token: str, userip: str):
        resp = requests.get(
            self.url,
            params={
                'method': 'oauth',
                'userip': userip,
                'format': 'json',
                'get_user_ticket': 'true',
                'oauth_token': oauth_token,
            },
            headers={
                'X-Ya-Service-Ticket': self.tvm_client.get_service_ticket_for(client_id=self.env.value)
            }
        )
        assert resp.status_code == 200, f'HTTP status {resp.status_code}, response body: {resp.text}'
        assert resp.json().get('error') == 'OK', resp.json()['error']
        return resp.json()

    def user_ticket_by_oauth(self, oauth_token: str, userip: str = None) -> str:
        userip = userip or get_qloud_ip()
        data = self.oauth(oauth_token, userip)
        return data['user_ticket']
