from datetime import datetime
from typing import Sequence, Collection

import requests

from .ticket_provider import TicketProdiver


class CalendarClient:
    def __init__(
        self,
        service_ticket_provider: TicketProdiver,
        user_ticket_provider: TicketProdiver,
        url: str,
        actor_uid: int,
    ):
        self.service_ticket_provider = service_ticket_provider
        self.user_ticket_provider = user_ticket_provider
        self.url = url
        self.actor_uid = actor_uid

    def _request(self, method, path, headers=None, **kwargs):
        headers = headers or {}
        headers['X-Ya-Service-Ticket'] = self.service_ticket_provider()
        headers['X-Ya-User-Ticket'] = self.user_ticket_provider()
        resp = requests.request(
            method,
            f'{self.url}/{path}',
            headers=headers,
            **kwargs
        )
        assert resp.status_code == 200, f'HTTP status {resp.status_code}, response body: {resp.text}'
        assert resp.json().get('error') is None, resp.json()['error']
        return resp.json()

    def get(self, path, params=None):
        return self._request('GET', path, params=params)

    def post(self, path, params=None, data=None, json=None):
        return self._request('POST', path, params=params, data=data, json=json)

    def get_resource_description(self, room_exchange_name: str = None):
        return self.get(
            'get-resource-description.json',
            params=dict(uid=self.actor_uid, email=room_exchange_name)
        )

    def get_resources_schedule(self, office_ids: Sequence[int], from_date: datetime, to_date: datetime, room_exchange_name: str = None):
        params = [
            ('office_id', str(oid)) for oid in office_ids
        ] + [
            ('uid', self.actor_uid),
            ('from', from_date.isoformat()),
            ('to', to_date.isoformat()),
        ]
        if room_exchange_name:
            params += [('email', room_exchange_name)]
        return self.get(
            'get-resources-schedule.json',
            params=params
        )

    def get_events_brief(self, event_ids: Collection[int]):
        return self.get(
            'get-events-brief.json',
            params=dict(
                uid=self.actor_uid,
                eventIds=','.join(map(str, event_ids))
            )
        )

    def get_event(self, event_id: int, tz='Europe/Moscow', date_format='zoned', **kwargs):
        return self.get(
            'get-event.json',
            params=dict(
                uid=self.actor_uid,
                eventId=event_id,
                tz=tz,
                dateFormat=date_format,
                **kwargs
            )
        )

    def get_events(self, from_date: datetime, to_date: datetime, target_uid: int = None, hide_absences: bool = False):
        return self.get(
            'get-events.json',
            params={
                'actorUid': self.actor_uid,
                'targetUid': target_uid or self.actor_uid,
                'hide_absences': hide_absences,
                'from': from_date.isoformat(),
                'to': to_date.isoformat(),
            }
        )

    def update_event(self, event_id: int, sequence: int, event_data, **kwargs):
        return self.post(
            'update-event.json',
            params=dict(
                id=event_id,
                uid=self.actor_uid,
                sequence=sequence,
                **kwargs
            ),
            json=event_data
        )
