import dataclasses
import json
from dataclasses import dataclass
from enum import Enum
from typing import List, Optional, Set, Any, Dict

import aiohttp


class UserType(Enum):
    YT = 'YT'
    BASIC = 'BASIC'


@dataclass
class CerberusUser:
    uid: int
    type: str
    login: str


@dataclass
class LocationKey:
    id: int
    type: str


@dataclass
class Resource:
    id: int
    type: str
    name: str
    active: bool
    location: Optional[LocationKey]
    info: Optional[dict]


@dataclass
class Grant:
    actions: Set[str]
    resource_type: str
    resource_id: Optional[int]


class SetEncoder(json.JSONEncoder):
    def default(self, obj: Any) -> Any:
        if isinstance(obj, set):
            return list(obj)
        else:
            return json.JSONEncoder.default(self, obj)


class CerberusClient:
    PUBLIC_CERBERUS_TESTING_URL = 'https://cerberus.testing.mail.yandex.net'
    CORP_CERBERUS_TESTING_URL = 'https://cerberus-yt.testing.mail.yandex.net'

    PUBLIC_CERBERUS_PROD_URL = 'http://cerberus-3.cerberus.production.cerberus-public.mail.stable.qloud-d.yandex.net'
    CORP_CERBERUS_PROD_URL = 'http://cerberus-1.cerberus.production.cerberus-yt.mail.stable.qloud-d.yandex.net'

    def __init__(self, connection_pool_size: int, url=PUBLIC_CERBERUS_TESTING_URL):
        self.url = url
        self.connector = aiohttp.TCPConnector(limit=connection_pool_size)

    async def __aenter__(self):
        return self

    async def __aexit__(self, exc_type, exc_val, exc_tb):
        await self.connector.close()

    async def _request(self, method: str, path: str, service_tvm_ticket: str, headers: Dict[str, str] = None, **kwargs):
        tvm_headers = {
            'X-Ya-Service-Ticket': service_tvm_ticket
        }

        headers = {**headers, **tvm_headers} if headers is not None else tvm_headers

        async with aiohttp.request(method=method, url=f'{self.url}/{path}', headers=headers, connector=self.connector,
                                   **kwargs) as resp:
            assert resp.status // 100 == 2, f'{resp}:\n{await resp.read()}'
            return await resp.read()

    async def get(self, path, service_tvm_ticket: str, params=None, headers: Dict[str, str] = None):
        return await self._request('GET', path, params=params, service_tvm_ticket=service_tvm_ticket, headers=headers)

    async def post(self, path, service_tvm_ticket: str, params=None, data=None, json=None, headers: Dict[str, str] = None):
        return await self._request('POST', path, params=params, data=data, json=json,
                                   service_tvm_ticket=service_tvm_ticket, headers=headers)

    async def add_users(self, skip_existing: bool, users: List[CerberusUser], service_tvm_ticket: str):
        return await self.post(
            'user/create/batch',
            service_tvm_ticket=service_tvm_ticket,
            params=dict(
                skip_existing=str(skip_existing).lower()
            ),
            json=[dataclasses.asdict(user) for user in users]
        )

    async def add_resources(self, skip_existing: bool, resources: List[Resource], service_tvm_ticket: str):
        return await self.post(
            'resource/add/bulk',
            service_tvm_ticket=service_tvm_ticket,
            params=dict(
                skip_existing=str(skip_existing).lower()
            ),
            json=[dataclasses.asdict(resource) for resource in resources]
        )

    async def add_user_grant(self, uid: int, grant: Grant, service_tvm_ticket: str):
        return await self.post(
            'grant/user',
            service_tvm_ticket=service_tvm_ticket,
            params=dict(
                uid=uid
            ),
            headers={'Content-Type': 'application/json'},
            data=json.dumps(dataclasses.asdict(grant), cls=SetEncoder)
        )
