#!/usr/bin/env python3
# encoding: utf-8

import logging
import os
from dataclasses import dataclass
from itertools import chain
from pathlib import Path
from typing import Sequence

import click
import requests

import yaml
from mail.python.qloud.component_info import QloudInfoProvider, Components
from mail.python.qloud.detail import dc_from_fqdn
from mail.python.qloud.nginx import update_configs, restore_configs

log = logging.getLogger(__name__)


@dataclass
class Upstream:
    name: str
    local_port: int
    components: Sequence[str]
    remote_port: int = None

    def __post_init__(self):
        if self.remote_port is None:
            self.remote_port = self.local_port


@dataclass
class Service:
    upstreams: Sequence[Upstream]

    def create_config_files(self, component_instances: Components, main_dc, output_config_dir: str):
        for upstream in self.upstreams:

            config_file = os.path.join(output_config_dir, upstream.name + '.include.new')
            remote_backends = '\n    '.join(
                chain.from_iterable(
                    (
                        f"server [{inst.ip}]:{upstream.remote_port};"
                        for inst in component_instances[component_name].instances
                        if dc_from_fqdn(inst.fqdn) == main_dc
                    )
                    for component_name in upstream.components
                    if component_name in component_instances
                )
            )

            with open(config_file, 'w') as f:
                # create upstream
                f.write(
                    f'upstream {upstream.name}-backend {{\n'
                    f'    least_conn;\n'
                    f'    {remote_backends}\n'
                    f'    server [::1]:{upstream.local_port} backup;\n'
                    f'}}\n'
                )

            log.info('Successfully created ' + config_file)
        return


def calendardb_master_dc(db_info_url):
    db_info = requests.get(db_info_url, verify=False).json()
    master = next(source for source in db_info['dataSources'] if source['isMaster'] and source['isAvailable'])
    return dc_from_fqdn(master['shortName'])


@click.option('-v', '--verbose', is_flag=True, default=False)
@click.option('-r', '--restore', is_flag=True, default=False, help="Restore upstream configs to their initial values. "
              "Also locks script from consequential runs unless|until `--unlock` option is used")
@click.option('--lock/--unlock', is_flag=True, default=None, help="Deny/allow consequential runs of script. Useful to "
              "prevent restored config from being modificated by crontab")
@click.option('--dry-run', is_flag=True, default=False, help="Check configuration changes to be made, but don't apply them")
@click.option('--config-path', default='/opt/nginx_upstream_gen_config.yml')
@click.command('calendar-nginx-gen-upstream')
def main(config_path: str, dry_run: bool, verbose: bool, restore: bool, lock: bool):
    logging.basicConfig(level=logging.DEBUG if verbose else logging.INFO)
    with open(config_path, 'r') as f:
        config = yaml.safe_load(f)
    nginx_config_dir = config['nginx_config_dir']

    lock_file = Path(config.get('lock_file', '/opt/nginx_upstream_gen_config.lock'))
    if lock is False:
        log.info(f'Removing lock-file {lock_file}')
        lock_file.exists() and lock_file.unlink()
    elif lock or restore:
        log.info(f'Locking script by lock-file {lock_file}')
        lock_file.touch()
        if lock:
            return
    elif lock_file.exists():
        log.error(f'Lock-file exists at path {lock_file}. Remove it with --unlock flag to proceed')
        return

    qloud_info = QloudInfoProvider(
        meta_file=config['meta_file'],
        api_endpoint=config['api_endpoint'],
    )
    upstreams = [
        upstream for upstream in
        (Upstream(**upstream_info) for upstream_info in config['upstreams'])
        if qloud_info.local_component() in upstream.components
    ]
    if not upstreams:
        log.info(f'There are no upstreams for component {qloud_info.local_component()}')
        return
    service = Service(upstreams)

    if restore:
        restore_configs(nginx_config_dir, [upstream.name for upstream in service.upstreams], dry_run=dry_run)
        return

    main_dc = calendardb_master_dc(db_info_url=config['db_info_endpoint'])
    if qloud_info.local_dc() == main_dc:
        log.info('Current server is already local to DB master, restoring')
        update_configs(nginx_config_dir, [], dry_run=dry_run)
        log.info(f'Removing lock-file {lock_file}')
        lock_file.exists() and lock_file.unlink()
    else:
        log.info(f'Going to setup proxying to servers in database-local dc {main_dc}')
        component_instances = qloud_info.components_info()
        log.debug(component_instances)
        service.create_config_files(
            component_instances=component_instances,
            main_dc=main_dc,
            output_config_dir=nginx_config_dir
        )
        update_configs(nginx_config_dir, [upstream.name for upstream in service.upstreams], dry_run=dry_run)
    log.info('All done')


if __name__ == '__main__':
    main()
