import json
from datetime import datetime, timedelta
import logging

import click
from typing import Any, Dict, Sequence

from mail.calendar.python.client.calendar_client import CalendarClient

from .helpers import calendar_client_options


log = logging.getLogger(__name__)

CALENDARETESTUSER_UID = 1120000000004717
TOP_MANAGER_UIDS = {
    'volozh': 1120000000000529,
    'bunina': 1120000000001062,
    'abovsky': 1120000000013893,
    'tigran': 1120000000000251,
}


def report_success():
    print('0;OK')


def report_warning(message: str):
    log.warning(message)
    print(f'1;{message}')


def report_error(message: str):
    log.error(message)
    message_header = message.split("\n")[0]
    print(f'2;{message_header}')


def report_viewable_events(events: Dict[str, Any]):
    def serialize_events(mgr_events: Sequence[Any], max_len: int = 5):
        result = ','.join(f"{event['id']}" for event in mgr_events[:max_len])
        if len(mgr_events) > max_len:
            result += ' and more'
        return result

    report_error(
        ';'.join(
            f'{top_manager} has {len(events)} viewable events ids: {serialize_events(events)}'
            for top_manager, events in events.items()
        )
    )


def event_could_be_viewable(event, top_manager):
    return event.get('othersCanView', False) or top_manager not in [att.get('login') for att in event.get('attendees', [])]


@click.command()
@click.option('--from-date', type=click.DateTime(), default=datetime.now().date().isoformat())
@click.option('--to-date', type=click.DateTime(), default=(datetime.now().date() + timedelta(days=30)).isoformat())
@calendar_client_options
def main_(cli: CalendarClient, from_date, to_date, **_):
    """Check events visibility of chosen top managers using auth creds of given actor"""
    total_viewable_events = {}
    for top_manager, target_uid in TOP_MANAGER_UIDS.items():
        events = cli.get_events(from_date=from_date, to_date=to_date, target_uid=target_uid, hide_absences=True).get('events')
        if events is None:
            return report_warning(f'No `events` field in response for {top_manager}: {json.dumps(events)}')
        illegally_viewable_events = [
            event for event in events
            if not event_could_be_viewable(event, top_manager)
        ]
        if illegally_viewable_events:
            total_viewable_events[top_manager] = illegally_viewable_events

    if total_viewable_events:
        report_viewable_events(total_viewable_events)
    else:
        report_success()


def main():
    """Wrap for main_ command that always provides juggler-compatible output"""
    try:
        logging.basicConfig(filename='/var/log/calendar/top-managers-visibility.log')
        main_()
    except Exception as e:
        report_warning(str(e))
