from enum import Enum
import functools
from typing import Type, TextIO

import click
from ticket_parser2.api.v1 import BlackboxClientId, TvmClient, TvmToolClientSettings

from mail.calendar.python.client.blackbox_client import BlackboxClient
from mail.calendar.python.client.calendar_client import CalendarClient
from mail.calendar.python.client.ticket_provider import (
    service_ticket_tvm_provider, fixed_ticket_provider, user_ticket_by_oauth_provider
)

from .calendar_properties import creds_from_propfile


def wraps_command(command):
    """
    Standard wrapping of function already decorated with `click.option` loses all already gathered command arguments.
    To save them through decorating, command magic field `__click__params__` must be assigned to wrapped function.
    """
    wrapper_assignments = functools.WRAPPER_ASSIGNMENTS + ('__click_params__',)
    return functools.wraps(command, assigned=wrapper_assignments)


def calendar_client_options(command):
    """Bind default set of calendar-cli options to a command; construct CalendarClient instance and pass it in"""
    @click.option('--env', type=EnumChoice(BlackboxClientId), default=BlackboxClientId.ProdYateam)
    @click.option('--host', default='http://localhost')
    @click.option('--actor-uid', type=int)
    @click.option('--actor-user-ticket', help='Use fixed user-ticket instead of getting one from `--actor-oauth-propfile`')
    @click.option('--actor-login', help='Used to get oauth token from `--actor-oauth-propfile`')
    @click.option('--actor-oauth-propfile', type=click.File(), default='/etc/yandex/calendar/secrets.properties')
    @click.option('--tvm-token-file', type=click.File())
    @wraps_command(command)
    def decorated_command(env: BlackboxClientId, host: str, actor_uid: int,
                          actor_user_ticket: str, actor_login: str, actor_oauth_propfile: TextIO,
                          tvm_token_file: TextIO,
                          **kwargs):
        if actor_uid and actor_login:
            raise RuntimeError('Both `actor-uid` and `actor-login` arguments are set; provide one')
        if actor_uid is None and actor_login is None:
            raise RuntimeError('Both `actor-uid` and `actor-login` arguments are not set; provide one')

        tvm_client = TvmClient(
            TvmToolClientSettings(
                self_alias='calendar',
                auth_token=tvm_token_file and tvm_token_file.read().strip(),
            )
        )
        actor_uid, user_ticket_provider = get_user_info(actor_login, actor_oauth_propfile, actor_uid,
                                                        actor_user_ticket, env, tvm_client)

        cli = CalendarClient(
            service_ticket_provider=service_ticket_tvm_provider(tvm_client, dst_alias='calendar'),
            user_ticket_provider=user_ticket_provider,
            url=host,
            actor_uid=actor_uid,
        )
        return command(cli=cli, tvm_client=tvm_client,
                       env=env, host=host, actor_uid=actor_uid, actor_user_ticket=actor_user_ticket,
                       actor_login=actor_login, actor_oauth_file=actor_oauth_propfile,
                       **kwargs)

    return decorated_command


def get_user_info(actor_login, actor_oauth_propfile, actor_uid, actor_user_ticket, env, tvm_client):
    if actor_user_ticket and actor_uid:
        user_ticket_provider = fixed_ticket_provider(actor_user_ticket)
    elif actor_login:
        user_creds = creds_from_propfile(env, actor_oauth_propfile, actor_login)
        actor_uid = user_creds.uid
        user_ticket_provider = user_ticket_by_oauth_provider(
            blackbox_client=BlackboxClient(env=env, tvm_client=tvm_client),
            oauth_token=user_creds.oauth
        )
    else:
        user_ticket_provider = None
    return actor_uid, user_ticket_provider


class EnumChoice(click.Choice):
    def __init__(self, enum_cls: Type[Enum], **kwargs):
        self.enum_cls = enum_cls
        super().__init__([choice.name for choice in self.enum_cls], **kwargs)

    def convert(self, value, *args, **kwargs):
        if isinstance(value, self.enum_cls):
            return value
        return self.enum_cls[super().convert(*args, **kwargs)]
