import asyncio
import time
from typing import Coroutine

from client.cerberus_client import CerberusClient, Grant
from client.db_client import DbClient, LayerPermissions
from client.db_client import DbSettings


def convert(layer_perms: LayerPermissions) -> Grant:
    return Grant(actions={action.name for action in layer_perms.perm.value}, resource_type='layer',
                 resource_id=layer_perms.layer_id)


async def main():
    batch_size = 300
    db_settings = DbSettings(
        host='',
        port=6432,
        dbname='',
        user='',
        password='',
        timeout_sec=5
    )
    cerberus_url = CerberusClient.PUBLIC_CERBERUS_TESTING_URL
    tvm_ticket = ''

    db = DbClient(db_settings, dry_run=False)
    await db.connect(pool_size=10)
    await db.ping()

    next_id = -1

    async with CerberusClient(url=cerberus_url, connection_pool_size=batch_size) as cerberus:
        print("Start migration")
        while next_id is not None:
            print(f'Fetch batch starting with id = {next_id}')
            fetch_start_time = time.time()
            batch = await db.select_layers_permissions(batch_size=batch_size, since_id=next_id)
            print(f'Received {len(batch)} layers permissions ({time.time() - fetch_start_time} seconds)')

            async def ignore_unexisting_user_error(coroutine: Coroutine):
                try:
                    return await coroutine
                except AssertionError as e:
                    if '"code":3' in str(e.args):
                        return None
                    else:
                        raise e

            push_start_time = time.time()
            await asyncio.gather(*[
                ignore_unexisting_user_error(
                    cerberus.add_user_grant(uid=perms.uid, grant=convert(perms), service_tvm_ticket=tvm_ticket)
                )
                for perms in batch
            ])
            push_end_time = time.time()
            print(f'Grants sent to cerberus ({push_end_time - push_start_time} seconds)')

            print(f'Batch migration complete ({push_end_time - fetch_start_time} seconds)')
            next_id = max((layers_perms.id for layers_perms in batch), default=None)
        print("Migration complete")


if __name__ == '__main__':
    asyncio.run(main())
