import asyncio
from typing import Optional

from mail.calendar.python.client.cerberus_client import CerberusClient, CerberusUser, UserType
from mail.calendar.python.client.db_client import DbClient, UserSettings
from mail.calendar.python.client.db_client import DbSettings


def parse_login(email: str) -> Optional[str]:
    parts = email.split('@')
    if len(parts) != 2:
        return None
    else:
        return parts[0]


def convert(settings: UserSettings, type: UserType) -> CerberusUser:
    login = settings.login or parse_login(settings.yandex_email)
    return CerberusUser(uid=settings.uid, type=type.value, login=login)


async def main():
    batch_size = 700
    user_type = UserType.BASIC
    db_settings = DbSettings(
        host='',
        port=6432,
        dbname='',
        user='',
        password='',
        timeout_sec=5
    )
    tvm_ticket = ''

    db = DbClient(db_settings, dry_run=False)
    cerberus = CerberusClient(url=CerberusClient.PUBLIC_CERBERUS_TESTING_URL)

    await db.connect(pool_size=10)
    await db.ping()

    next_uid = -1

    print("Start migration")
    while next_uid is not None:
        print(f'Fetch batch starting with uid = {next_uid}')
        batch = await db.select_user_settings(batch_size=batch_size, since_id=next_uid)
        print(f'Received {len(batch)} users')
        cerberus_users = [convert(settings, type=user_type) for settings in batch]
        await cerberus.add_users(users=cerberus_users, skip_existing=True, service_tvm_ticket=tvm_ticket)
        print("Batch migration complete")
        next_uid = max((user.uid for user in batch), default=None)
    print("Migration complete")


if __name__ == '__main__':
    asyncio.run(main())
