import json
from datetime import datetime, timedelta

import click

from mail.calendar.python.client.calendar_client import CalendarClient

from .helpers import calendar_client_options


@click.command()
@click.option('--remove-room-email', required=True)
@click.option('--add-room-email', required=True)
@click.option('--from-date', type=click.DateTime(), default=datetime.now().date().isoformat())
@click.option('--to-date', type=click.DateTime(), default=(datetime.now().date() + timedelta(days=500)).isoformat())
@calendar_client_options
def main(cli: CalendarClient, remove_room_email, add_room_email, from_date, to_date, **_):
    """For all events with `--remove-room-email` for given date range, replace that room with `--add-room-email`"""
    room_to_add = cli.get_resource_description(add_room_email)
    room_to_remove = cli.get_resource_description(remove_room_email)
    assert room_to_add['officeId'] == room_to_remove['officeId'], 'Rooms must be in same office'
    room_schedule = cli.get_resources_schedule([room_to_remove['officeId']], from_date, to_date, remove_room_email)
    event_infos = room_schedule['offices'][0]['resources'][0]['events']
    for event in event_infos:
        if 'eventId' not in event:
            continue
        print(f'Event brief: {event}')
        event_id = event['eventId']
        event = cli.get_event(event_id, instanceStartTs=event['start'])
        if input(f'''
Going to replace room for event:
{json.dumps(event, indent=4, ensure_ascii=False)}
Press [nN] to skip this event, any other key to continue:''') in ('n', 'N'):
            continue
        replace_room(cli, event, add_room_email, remove_room_email)


def replace_room(cli, event, add_room_email, remove_room_email):
    event_id = event['id']
    update_info = {
        key: event[key] for key in [
            'type', 'startTs', 'endTs', 'name', 'location', 'description', 'isAllDay',
            # 'availability',
            'notifications', 'participantsCanInvite', 'participantsCanEdit', 'othersCanView'
        ]
    }
    update_info['organizer'] = event['organizer']['email']
    if 'repetition' in event:
        update_info['repetition'] = event['repetition']
    attendees = [user['email'] for user in event['attendees']]
    rooms = set(room['email'] for room in event['resources'])
    rooms.remove(remove_room_email)
    rooms.add(add_room_email)
    update_info['attendees'] = attendees + list(rooms)
    print(cli.update_event(
        event_id,
        sequence=event['sequence'],
        event_data=update_info,
        applyToFuture="false" if event['isRecurrence'] else "true",
        instanceStartTs=event['instanceStartTs'],
        # instanceStartTs=event['startTs'] if event['isRecurrence'] else event['instanceStartTs'],
    ))
    input()
