// should be stored in ZK as ews-desync at dynamic-monitoring.zk-root-path
// see EwsDesyncDynamicMonitoring

// uncomment and fill for manual run
// var scriptSetup = new ru.yandex.calendar.monitoring.EwsDesyncDynamicMonitoring.DesyncMonitoringSetup(
//    14, Cf.list(), Cf.list(), 30);
// var previousLaunchData = Option.of(Cf.map());

importPackage(org.joda.time);
importPackage(ru.yandex.misc.time);
importPackage(ru.yandex.calendar.logic.event.avail);
importPackage(ru.yandex.calendar.logic.event);

var daysInFuture = scriptSetup.getDaysInFuture();
var logins = scriptSetup.getLogins();
var mute = Tuple2List.fromPairs(scriptSetup.getMute().toArray());

var now = Instant.now();

var interval = new InstantInterval(now, now.plus(Duration.standardDays(daysInFuture)));

var emails = logins.filterMap(function(login) {return Option.x(userManager.getYtUserEmailByLogin(login));});

var logger = getField(ewsComparator, 'logger');

var critDesync = Tuple2.tuple('crit', 2);
var mediumDesync = Tuple2.tuple('medium', 1);
var unknownDesync = Tuple2.tuple('unknown', 0);

function getDesync(r, exchangeEmail, exchangeUnique) {
    var externalId = getField(r, 'externalIdO');
    var startTs = getField(r, 'startTs');
    var endTs = getField(r, 'endTs');
    var name = getField(r, 'name');
    var recurrenceIdO = getField(r, 'recurrenceIdO');
    var attendeesDecisions = getField(r, 'attendeesDecisions');

    var ewsOrganizerO = getField(r, 'ewsOrganizerO');

    var subjectDecision = getField(r, 'subjectDecision');
    var availability = getField(r, 'availability');

    var result = {};

    result.reason = '';
    result.desyncLevel = unknownDesync;

    function setDesyncLevel(newLevel) {
        if (newLevel._2 > result.desyncLevel._2) {
            result.desyncLevel = newLevel;
        }

        if (newLevel == critDesync) {
            result.reason += ' <--- this was critical'
        }
    }

    result.externalId = externalId.getOrElse('-') + (recurrenceIdO.isPresent() ? '/' + recurrenceIdO.get() : '');

    if (!externalId.isPresent()) {
        result.reason = 'no external id';
        setDesyncLevel(critDesync);
        return result;
    }

    result.ewsOrganizer = ewsOrganizerO.getOrElse('-');

    var withSameExternalId = exchangeUnique.count(function (k) {
        return externalId.equals(getField(k, 'externalIdO'))});

    if (withSameExternalId == 0) {
        result.reason = 'Does not exists in exchange. Start: ' + startTs;

        var isNonEwsGap = result.externalId.endsWith('gap.yandex-team.ru') && !ewsOrganizerO.isPresent();

        setDesyncLevel(isNonEwsGap ? mediumDesync : critDesync);
        return result;
    }

    var exchangeKeyO = exchangeUnique.filter(function (k) {
        return externalId.equals(getField(k, 'externalIdO'))
            && startTs.equals(getField(k, 'startTs'))
            && endTs.equals(getField(k, 'endTs'))}).toList().firstO();

    if (!exchangeKeyO.isPresent()) {
        var ks = exchangeUnique.filter(function (k) {
            return externalId.equals(getField(k, 'externalIdO'))}).toList();

        result.reason = 'Start or end time mismatch:\n' + startTs + ' - ' + endTs + ' in calendar vs\n';
        ks.forEach(function(k) {
            result.reason += 'may be:' + getField(k, 'startTs') + ' - ' +  getField(k, 'endTs') + '\n';
        });
        result.reason += 'in exchange';

        setDesyncLevel(critDesync);
        return result;
    }

    if (!name.equals(getField(exchangeKeyO.get(), 'name'))) {
        var nameInExchange = getField(exchangeKeyO.get(), 'name');
        result.reason = 'Name mismatch:\n' + name + ' in calendar vs\n' + nameInExchange + ' in exchange';

        var normalize = org.apache.commons.lang3.StringUtils.deleteWhitespace;

        setDesyncLevel(normalize(nameInExchange).contains(normalize(name))
            || normalize(name).contains(normalize(nameInExchange)) ? mediumDesync : critDesync);
    }

    var cAttendees = attendeesDecisions;
    var eAttendees = getField(exchangeKeyO.get(), 'attendeesDecisions');

    function toDecisionMap(attendees) {
        return attendees.mapValues(function (dvs) {return dvs.getDecision()})
    }

    if (!toDecisionMap(cAttendees).equals(toDecisionMap(eAttendees))) {
        var keys = cAttendees.keys().plus(eAttendees.keys()).unique();
        var mismatch = keys.filter(function(key) {
            cDecisionO = cAttendees.getO(key).map(function (dvs) {return dvs.getDecision()});
            eDecisionO = eAttendees.getO(key).map(function (dvs) {return dvs.getDecision()});

            if (cDecisionO.equals(eDecisionO)) {
                return false;
            }

            return !cAttendees.getO(key).map(function (dvs) {
                return dvs.isFromExchangeOrMail()
            }).isSome(true);

        });
        if (result.reason) {
            result.reason += '\n';
        }

        result.reason += 'Decisions mismatch for ' + mismatch;
        setDesyncLevel(mediumDesync);
    }

    var subjectDecisionInExchange = getField(exchangeKeyO.get(), 'subjectDecision');

    if (!subjectDecision.equals(subjectDecisionInExchange)) {
        if (result.reason) {
            result.reason += '\n';
        }
        result.reason += 'Subject decision mismatch: ' + subjectDecision + ' in calendar vs ' + subjectDecisionInExchange + ' in exchange';
        setDesyncLevel(mediumDesync);
    }

    var availabilityInExchange = getField(exchangeKeyO.get(), 'availability');

    if (!availability.equals(availabilityInExchange)) {
        if (result.reason) {
            result.reason += '\n';
        }
        result.reason += 'Subject availability mismatch: ' + availability + ' in calendar vs ' + availabilityInExchange + ' in exchange';

        var availableSomeWhere = Cf.list(availability, availabilityInExchange).contains(Availability.AVAILABLE);

        setDesyncLevel(availableSomeWhere ? critDesync : mediumDesync);
    }

    var ewsOrganizerOInExchange = getField(exchangeKeyO.get(), 'ewsOrganizerO');

    var isOrganizerInExchange = ewsOrganizerOInExchange.isPresent();
    var isEwsOrganizerInCalendar = ewsOrganizerO.isSome(exchangeEmail);

    if (isOrganizerInExchange != isEwsOrganizerInCalendar) {
        if (result.reason) {
            result.reason += '\n';
        }
        // XXX needs to be fixed
        var isMeeting = attendeesDecisions.keys().size() > 1;
        result.reason += 'Ews organizer status mismatch: ' + ewsOrganizerO + ' in calendar vs ' + ewsOrganizerOInExchange + ' in exchange';
        setDesyncLevel(isMeeting ? critDesync : mediumDesync);
    }

    var sequenceInCalendar = getField(r, 'sequence');
    var sequenceInExchange = getField(exchangeKeyO.get(), 'sequence');

    if (sequenceInCalendar < sequenceInExchange) {
        if (result.reason) {
            result.reason += '\n';
        }
        result.reason += 'Sequence mismatch: ' + sequenceInCalendar + ' in calendar less then ' + sequenceInExchange + ' in exchange';
        setDesyncLevel(critDesync);
    }

    if (result.desyncLevel == unknownDesync) {
        result.reason = 'unknown desync';
        setDesyncLevel(critDesync);
    }

    return result;
}

function listExchangeOnlyEvents(r, calendarUnique) {
    var externalId = getField(r, 'externalIdO');
    var recurrenceIdO = getField(r, 'recurrenceIdO');
    var startTs = getField(r, 'startTs');
    var endTs = getField(r, 'endTs');
    var exchangeIdO = getField(r, 'exchangeId');

    var withSameExternalId = calendarUnique.count(function (k) {
        return externalId.equals(getField(k, 'externalIdO'))});

    if (withSameExternalId == 0) {
        var result = {};

        result.reason = 'Exists only in exchange. Start: ' + startTs + ', externalId: ' + externalId + ', exchangeId: ' + exchangeIdO.orElse('unknown');
        var now = Instant.now();
        var ongoing = (now.isAfter(startTs) && now.isBefore(endTs) && externalId.isPresent()) ;

        result.desyncLevel = ongoing ? mediumDesync : critDesync;
        result.externalId = externalId.getOrElse('-') + (recurrenceIdO.isPresent() ? '/' + recurrenceIdO.get() : '');

        return Option.of(result);
    } else {
        return Option.empty();
    }

}

function analizeUniqueEvents(userCompareResult) {
    var exchangeUnique = userCompareResult.getExchangeUniqueKeys();
    var calendarUnique = userCompareResult.getCalendarUniqueKeys();
    var exchangeEmail = userCompareResult.getExchangeEmail();

    var desyncs = calendarUnique.map(function(r) {
        return getDesync(r, exchangeEmail, exchangeUnique);
    }).plus(exchangeUnique.filterMap(function(r) {
        return listExchangeOnlyEvents(r, calendarUnique);
    }));

    var result = {};
    result.email = userCompareResult.getExchangeEmail();
    result.desyncs = desyncs;

    return result;
}

function makeFullReport(allDesyncs) {
    var message = '';
    allDesyncs.forEach(function(login, desyncs) {
        message += '\n**** ' + login + ' ****\n';
        message += makeMessage(desyncs)
    });
    return message;
}

function makeMessage(desyncs) {
    var message = '';
    desyncs.forEach(function(desync) {
        message += 'Ews Organizer: ' + desync.ewsOrganizer + '\n';
        message += 'ExternalId: ' + desync.externalId + '\n';
        message += 'Desync level: ' + desync.desyncLevel + '\n';
        message += 'Reason: ' + desync.reason + '\n';
        message += '----' + '\n';
    });
    return message;
}

function alertsToMap(listOfTuples) {
    return listOfTuples.toTuple2List(function (t) {return t;}).toMap().filterValues(function (ids) {return ids.isNotEmpty()});
}

function filterMonitor(allUsersDesync, level, idOnly) {
    return alertsToMap(allUsersDesync.map(function (userDesync) {
        var login = userDesync.email.getLocalPart();
        var desyncs = userDesync.desyncs.filterMap(function (desync) {
            if (mute.contains(Tuple2.tuple(login, desync.externalId))) {
                return Option.empty();
            }
            return Option.when(desync.desyncLevel == level, idOnly ? desync.externalId : desync);
        });
        return Tuple2.tuple(login, desyncs);
    }));
}

var compareResult = emails.map(function(email) {
    logger.info('start comparing for ' + email.toString());
    return ewsComparator.compare(email, interval);
});

var allUsersDesync = compareResult.map(function(userCompareResult) {return analizeUniqueEvents(userCompareResult);});

var common = compareResult.map(function (c) {return c.getCommonKeys()}).flatten().size();
var warn = filterMonitor(allUsersDesync, mediumDesync, true);
var crit = filterMonitor(allUsersDesync, critDesync, true);
var critConfirmed = alertsToMap(crit.mapEntries(function(login, ids) {
    var previousCritIds = previousLaunchData.getOrElse(Cf.map()).getOrElse(login, Cf.list()).unique();
    return Tuple2.tuple(login, ids.unique().intersect(previousCritIds).toList())
}));

var scriptResult = new ru.yandex.calendar.monitoring.EwsDesyncDynamicMonitoring.DesyncMonitoringResult(common, warn, crit, critConfirmed);

var fullReport = makeFullReport(filterMonitor(allUsersDesync, critConfirmed.isEmpty() ? mediumDesync : critDesync, false));
logger.info('Desync full report.\n' + fullReport);
