import logging

import jsonschema
from aiohttp import web, web_exceptions
from aiohttp.hdrs import METH_POST, METH_PUT
from aiohttp.web_request import Request
from aiohttp.web_response import Response
from mail.callmeback.callmeback.detail.errors import BadRequest, InvalidEntityData
from mail.callmeback.callmeback.detail.http_helpers import jsend
from mail.callmeback.callmeback.swagger import SWAGGER_SCHEMA
from mail.python.theatre.logging.request_id import set_request_id

log = logging.getLogger(__name__)


@web.middleware
async def request_id_middleware(request: Request, handler) -> Response:
    with set_request_id(prefix='httpreq', value=request.headers.getone('X-Request-Id', None)) as rid:
        middle: Response = await handler(request)
        middle.headers.add('X-Request-Id', rid)
        return middle


@web.middleware
async def jsend_middleware(request, handler) -> Response:
    try:
        return jsend.success(await handler(request))
    except BadRequest as e:
        log.error('Request failed due to error <%s>, fail_data: %s', str(e), e.fail_data)
        return jsend.fail(str(e), status=e.fail_status, data=e.fail_data)
    except NotImplementedError:
        return jsend.not_implemented()
    except web_exceptions.HTTPException:
        raise
    except Exception as e:
        log.exception(e)
        return jsend.error(str(e))


TYPE_CONVERTER = {
    'integer': int,
    # XXX: can have bad precision and validation will fail
    'number': float,
    'array': lambda v: v.split('|'),
    'boolean': lambda v: v.lower() in ['true', 'yes', '1'],
}


def validate(arg, schema):
    # TODO support deep conversion
    try:
        arg = TYPE_CONVERTER.get(schema['type'], lambda v: v)(arg)
    except Exception:
        # Wanna have nice jsonschema validation error
        pass
    return jsonschema.validate(arg, schema, format_checker=jsonschema.FormatChecker())


@web.middleware  # noqa: C901
async def schema_validation_middleware(request: web.Request, handler):
    """Request url path before path variable substitutions, e.g.: /v1/event/{group_key}/{event_key}"""
    url_resource = request.match_info.route.resource
    if url_resource is None:
        # Nothing to validate, probably resource does not exist
        return await handler(request)

    url_path = url_resource.canonical
    try:
        path_schema = SWAGGER_SCHEMA['paths'][url_path]
    except KeyError:
        # XXX Pass silently if no schema exists
        return await handler(request)
    parameters = path_schema[request.method.lower()]['parameters']

    for args, arg_getter in [
        (filter(lambda x: x.get('in') == 'path', parameters), request.match_info.get),
        (filter(lambda x: x.get('in') == 'query', parameters), request.query.get),
    ]:
        for arg_node in args:
            arg = arg_getter(arg_node['name'])
            if arg:
                schema = arg_node.get('schema', arg_node)
                try:
                    validate(arg, schema)
                except jsonschema.ValidationError as e:
                    raise InvalidEntityData(str(e), fail_data={
                        'schema': schema,
                        'absolute_path': e.absolute_path,
                        'absolute_schema_path': e.absolute_schema_path,
                    })

    # Validation of post-data
    if request.method in [METH_POST, METH_PUT]:
        bodies = list(filter(lambda x: x.get('in') == 'body', parameters))
        if bodies:
            # We have schema for body part of request, validate it
            body_schema = bodies[0]['schema']
            try:
                validate(await request.json(), body_schema)
            except jsonschema.ValidationError as e:
                raise InvalidEntityData(str(e), fail_data={
                    'schema': body_schema,
                    'absolute_path': e.absolute_path,
                    'absolute_schema_path': e.absolute_schema_path,
                })

    resp = await handler(request)
    return resp
