import logging
import os
import ujson

from dataclasses import dataclass
from yarl import URL

from mail.callmeback.callmeback.detail.errors import BadRequest
from mail.python.theatre.app.settings import TvmSettings
from mail.python.theatre.roles import HttpPooled

log = logging.getLogger(__name__)


@dataclass
class TicketInfo:
    src: int
    dst: int
    scopes: list
    debug_string: str
    logging_string: str

    @staticmethod
    def from_dict(data):
        return TicketInfo(
            src=int(data.get('src')),
            dst=int(data.get('dst')),
            scopes=data.get('scopes'),
            debug_string=data.get('debug_string'),
            logging_string=data.get('logging_string'),
        )


# TODO :: look at sendr-qtools.sendr_tvm
# https://github.yandex-team.ru/sendr/sendr-qtools/tree/master/src/sendr_tvm
class TvmServiceTickets(HttpPooled):
    @staticmethod
    def from_conf(settings: TvmSettings):
        local_token = os.environ.get('QLOUD_TVM_TOKEN')
        if settings.client_id and local_token:
            return TvmServiceTickets(
                local_token,
                settings.client_id,
                tvm_daemon_url=settings.daemon_url,
                **settings.http_pool.as_dict()
            )

    def __init__(self, local_token: str, client_id: int, tvm_daemon_url: str, **kwargs):
        super(TvmServiceTickets, self).__init__(**kwargs)
        self._local_token = local_token
        self._checksrv_url = URL(tvm_daemon_url + '/tvm/checksrv')
        self._get_tickets_url = URL(tvm_daemon_url + '/tvm/tickets')
        self._client_id = client_id

    async def parse(self, service_ticket) -> TicketInfo:
        async with self.http_session.get(
                url=self._checksrv_url,
                headers={
                    'Authorization': self._local_token,
                    'X-Ya-Service-Ticket': service_ticket,
                }
        ) as response:
            if response.status != 200:
                log.error(await response.text())
                raise RuntimeError('TVM problems')
            data = await response.json(loads=ujson.loads)
            if 'error' in data:
                log.error('Error: %s ticket: %s', data['error'], data['logging_string'])
                raise BadRequest(data['error'])
            if data['dst'] != self._client_id:
                log.error('Wrong dst in ticket: %s', data['logging_string'])
                raise BadRequest('Wrong dst in TVM ticket')
            return TicketInfo.from_dict(data)

    async def get(self, dst_client_id: int):
        async with self.http_session.get(
                url=self._get_tickets_url,
                params={'dsts': dst_client_id},
                headers={'Authorization': self._local_token},
        ) as response:
            if response.status != 200:
                log.error(await response.text())
                raise RuntimeError('TVM problems')
            data = await response.json(loads=ujson.loads)
            if 'error' in data:
                log.error('Error: %s', data['error'])
                raise BadRequest(data['error'])
            if len(data) != 1:
                log.error('Single element in TVM response expected, got: %s', data)
                raise BadRequest('Strange tvm stuff')
            ticket_data = list(data.values())[0]
            if 'error' in ticket_data:
                log.error('Error: %s ticket: %s', ticket_data['error'], ticket_data['logging_string'])
                raise BadRequest(ticket_data['error'])
            return ticket_data['ticket']
