import logging
from datetime import timedelta

log = logging.getLogger(__name__)


class ConstantRetryStrategy():
    def __init__(self, base_delay: timedelta):
        self._base_delay = base_delay

    def get_retry_delay(self, retry_number: int):
        return self._base_delay

    @staticmethod
    def from_dict(params):
        base_delay = timedelta(seconds=params['base_delay'])
        return ConstantRetryStrategy(base_delay=base_delay)


class ExponentialRetryStrategy():
    def __init__(self, base_delay: timedelta, growth_rate: float):
        self._base_delay = base_delay
        self._growth_rate = growth_rate

    def get_retry_delay(self, retry_number: int):
        return self._base_delay * (self._growth_rate ** (retry_number - 1))

    @staticmethod
    def from_dict(params):
        base_delay = timedelta(seconds=params['base_delay'])
        growth_rate = float(params['delay_growth_rate'])
        return ExponentialRetryStrategy(base_delay=base_delay, growth_rate=growth_rate)


class RetryStrategies:
    @staticmethod
    def from_dict(params):
        strategy = params.get('strategy')
        if strategy is None:
            return None
        elif strategy == 'constant':
            return ConstantRetryStrategy.from_dict(params)
        elif strategy == 'exponential':
            return ExponentialRetryStrategy.from_dict(params)
        else:
            raise RuntimeError(f'Unknown retry strategy: {strategy}')


class RetriesLimit:
    def __init__(self, stop_after_delay: timedelta):
        self._stop_after_delay = stop_after_delay

    @property
    def stop_after_delay(self):
        return self._stop_after_delay

    @staticmethod
    def from_dict(params):
        stop_after_delay = params.get('stop_after_delay')
        if stop_after_delay is not None:
            return RetriesLimit(stop_after_delay=timedelta(seconds=stop_after_delay))
        return None


class QuickRetries:
    def __init__(self, retries_count: int):
        self._retries_count = retries_count

    @property
    def retries_count(self):
        return self._retries_count

    @staticmethod
    def from_dict(params):
        retries_count = params.get('quick_retries_count')
        if retries_count is not None:
            return QuickRetries(retries_count=int(retries_count))
        return None


class RetryParams:
    def __init__(self, config):
        self._config = config

    @staticmethod
    def check_params(params):
        RetryStrategies.from_dict(params)
        RetriesLimit.from_dict(params)
        QuickRetries.from_dict(params)

    def _parse_params(self, func, params, default):
        parsed_param = None
        try:
            parsed_param = func(params)
        except Exception:
            log.warning(f'Bad retry_params: {params}')
        if parsed_param is None:
            parsed_param = default
        return parsed_param

    def get_retry_delay(self, params: dict, retry_number: int):
        retry_strategy = self._parse_params(RetryStrategies.from_dict, params, self._config.retry_strategy)
        return retry_strategy.get_retry_delay(retry_number)

    def get_stop_after_delay(self, params: dict):
        retries_limit = self._parse_params(RetriesLimit.from_dict, params, self._config.retries_limit)
        return retries_limit.stop_after_delay

    def get_quick_retries_count(self, params: dict):
        quick_retries = self._parse_params(QuickRetries.from_dict, params, self._config.quick_retries)
        return quick_retries.retries_count
