import logging

import ujson
from aiohttp.web import Response

from mail.callmeback.callmeback.detail.http_helpers.tvm import TvmServiceTickets
from mail.callmeback.callmeback.stages.api.props.add_reminder import AddReminder
from mail.callmeback.callmeback.stages.api.props.event_api import EventApi
from mail.callmeback.callmeback.stages.api.props.event_buffer import EventBuffer
from mail.callmeback.callmeback.stages.api.settings import Settings
from mail.python.theatre.roles.director import Director as DirectorBase
from mail.python.theatre.app.roles.db_multihost_pool import DbMultihostPool
from .force_insert import InsertionForcer
from .worker_monitor import ActiveWorkerMonitor
from .worker_notifier import WorkerNotifier

log = logging.getLogger(__name__)


class Director(DirectorBase):
    add_reminder: AddReminder
    event_api: EventApi

    def __init__(self, settings: Settings, pool: DbMultihostPool):
        worker_mon = ActiveWorkerMonitor(pool=pool, settings=settings.worker_monitor)
        worker_notifier = WorkerNotifier(workers_monitor=worker_mon, settings=settings.worker_notifier)
        buffer = EventBuffer(
            pool=pool,
            notifier=worker_notifier,
            worker_monitor=worker_mon,
            settings=settings.event_buffer,
        )
        insert_by_clock = InsertionForcer(buffer=buffer, settings=settings.insertion_forcer)
        tasks = [pool.poller, worker_mon, worker_notifier, insert_by_clock]
        tvm_tickets = TvmServiceTickets.from_conf(settings.tvm)
        if tvm_tickets:
            tasks.append(tvm_tickets)
        super(Director, self).__init__(tasks=tasks)

        self.add_reminder = AddReminder(buffer=buffer, tvm_tickets=tvm_tickets)
        self.event_api = EventApi(pool=pool, tvm_tickets=tvm_tickets, settings=settings.event_api)

    async def unistat_handler(self, _):
        return Response(
            text=ujson.dumps(self.metrics()),
            content_type='application/json'
        )
