import asyncio

from datetime import timedelta
from aiohttp import web
import click

from mail.callmeback.callmeback.detail.http_helpers.middleware import (
    request_id_middleware, jsend_middleware, schema_validation_middleware
)
from mail.callmeback.callmeback.stages.api.libretto.director import Director
from mail.callmeback.callmeback.stages.api.props.add_reminder import AddReminder
from mail.callmeback.callmeback.stages.api.props.event_api import EventApi
from mail.callmeback.callmeback.swagger import SWAGGER_SCHEMA
from mail.python.theatre.stages.base import Stage
from mail.python.theatre.app.roles.db_multihost_pool import DbMultihostPool
from .settings import Settings


class ApiStage(Stage):
    def __init__(self, env: str):
        super(ApiStage, self).__init__(env=env, name='callmeback-api', settings=Settings)

    async def create_app(self):
        pool = await DbMultihostPool(
            db_settings=self.settings.db,
            db_host_status_poll_time=timedelta(seconds=30),
        ).create_pool()
        director = Director(self.settings, pool)
        self.app.update(pool=pool, director=director)
        await director.start()

        async def shutdown(_):
            await director.stop()
            await pool.close()

        self.app.on_shutdown.append(shutdown)
        self.setup_routes(add_reminder=director.add_reminder, event_api=director.event_api, director=director)

        return self.app

    def setup_routes(self, add_reminder: AddReminder, event_api: EventApi, director: Director):
        v1app = web.Application(
            middlewares=[request_id_middleware, jsend_middleware, schema_validation_middleware]
        )
        v1app.router.add_post('/event/add/{group_key}/{event_key}', add_reminder, name='add_reminder')
        event_api.bind_routes(v1app)
        self.app.add_subapp('/v1', v1app)

        self.app.router.add_get('/unistat', director.unistat_handler, name='unistat')
        self.app.router.add_get('/ping', lambda _: web.Response(text='pong'))
        setup_swagger(self.app, SWAGGER_SCHEMA)


def setup_swagger(app: web.Application, schema):
    async def schema_handler(_):
        return web.json_response(schema, headers={
            'Access-Control-Allow-Origin': '*',
            'Access-Control-Allow-Methods': 'GET',
            'Access-Control-Allow-Headers': 'Content-Type',
        })

    swagger_app = web.Application()
    swagger_app.router.add_get('/swagger.json', schema_handler)
    app.add_subapp('/schema', swagger_app)


@click.command()
@click.option('--host', default='::0', help='IP addr to bind to')
@click.option('--port', default=80, help='HTTP port to listen')
@click.option('--env', help='Environment name')
def run_main(host, port, env=None):
    import uvloop
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())

    stage = ApiStage(env=env)
    stage.run(host=host, port=port)


if __name__ == '__main__':
    run_main()
