from datetime import datetime

import ujson
from aiohttp import web

from mail.callmeback.callmeback.detail.validation import parse_datetime, check_context, validated_url
from mail.callmeback.callmeback.detail.http_helpers.tvm import TvmServiceTickets
from mail.callmeback.callmeback.stages.api.props.queries import Queries, ExternalEventInfo
from mail.callmeback.callmeback.stages.api.settings import EventApiSettings
from mail.python.theatre.app.roles.db_multihost_pool import DbMultihostPool


class EventApi:
    def __init__(self, pool: DbMultihostPool, tvm_tickets: TvmServiceTickets, settings: EventApiSettings):
        self._queries = Queries(pool)
        self._settings = settings
        self._tvm_tickets = tvm_tickets

    def bind_routes(self, app: web.Application):
        app.router.add_post('/event/cancel/{group_key}/{event_key}', self.cancel)
        app.router.add_post('/event/delete/{group_key}/{event_key}', self.delete)
        app.router.add_post('/event/update/{group_key}/{event_key}', self.update)
        app.router.add_get("/find/{group_key}", self.find)
        app.router.add_get('/event/{group_key}/{event_key}', self.get)
        app.router.add_get('/event/{group_key}', self.list)

    async def _event_info(self, request: web.Request):
        tvm_ticket = request.headers.getone('X-Ya-Service-Ticket', None)
        return ExternalEventInfo(
            client_id=self._tvm_tickets and tvm_ticket and (await self._tvm_tickets.parse(tvm_ticket)).src,
            group_key=request.match_info.get('group_key'),
            event_key=request.match_info.get('event_key')
        )

    async def cancel(self, request: web.Request):
        return await self._queries.cancel(
            event_info=await self._event_info(request),
            backoff_sett=self._settings.cancel,
        )

    async def delete(self, request: web.Request):
        return await self._queries.delete(
            event_info=await self._event_info(request),
            min_run_at=datetime.now() + self._settings.min_delete_run_at,
            force=bool(ujson.loads(request.query.get('force', '0').lower())),
            backoff_sett=self._settings.delete,
        )

    async def update(self, request: web.Request):
        body = await request.text()
        data = ujson.loads(body) if body else {}

        if 'run_at' in data:
            data['run_at'] = parse_datetime(data['run_at'], 'run_at')
        if 'cb_url' in data:
            data['cb_url'] = validated_url(data['cb_url'], 'cb_url')
        if 'context' in data:
            check_context(data['context'])
            data['context'] = ujson.dumps(data['context'])
        return await self._queries.update(
            event_info=await self._event_info(request),
            backoff_sett=self._settings.delete,
            ensure_pending=bool(ujson.loads(request.query.get('ensure_pending', '0').lower())),
            **data
        )

    async def get(self, request: web.Request):
        return await self._queries.get(
            event_info=await self._event_info(request),
            backoff_sett=self._settings.get,
        )

    async def list(self, request: web.Request):
        events_info = await self._event_info(request)
        return await self._queries.list(
            client_id=events_info.client_id,
            group_key=events_info.group_key,
            backoff_sett=self._settings.list,
        )

    async def find(self, request: web.Request):
        events_info = await self._event_info(request)
        return await self._queries.list_by_prefix(
            client_id=events_info.client_id,
            group_key=events_info.group_key,
            event_key_prefix=request.query.get("event_key_prefix", ""),
            backoff_sett=self._settings.get,
        )
