import logging
from datetime import datetime
from typing import Dict, Any

import asyncpg
import backoff
import sqlalchemy as sa
import ujson
from asyncpg import Record
from asyncpgsa import compile_query
from dataclasses import dataclass, asdict

from mail.callmeback.callmeback.detail.errors import EntityDoesNotExist, InvalidEntityData, InvalidEntityCondition
from mail.python.theatre.app.settings import BackoffSettings
from mail.python.theatre.app.roles.db_multihost_pool import DbMultihostPool

log = logging.getLogger(__name__)


@dataclass
class ExternalEventInfo:
    client_id: int = None
    group_key: str = None
    event_key: str = None


class Queries:
    CANCEL_Q = """
        -- cancel_event
        select * from code.cancel_event(:client_id, :group_key, :event_key)
    """

    DELETE_Q = """
        -- delete_event
        select * from code.delete_event(:client_id, :group_key, :event_key, :min_run_at, :force)
    """

    UPDATE_Q = """
        -- update_event
        select * from code.update_event(:client_id, :group_key, :event_key, :run_at, :cb_url, :context, :ensure_pending)
    """

    GET_Q = """
        -- get_event
        SELECT *
          FROM reminders.events
         WHERE COALESCE(owner_client_id, -1) = COALESCE(cast(:client_id as bigint), -1)
           AND group_key = :group_key
           AND event_key = :event_key
    """

    LIST_Q = """
        -- list_events
        SELECT *
          FROM reminders.events
         WHERE COALESCE(owner_client_id, -1) = COALESCE(cast(:client_id as bigint), -1)
           AND group_key = :group_key
    """

    FIND_Q = """
            -- list_by_prefix
            SELECT *
              FROM reminders.events
             WHERE COALESCE(owner_client_id, -1) = COALESCE(cast(:client_id as bigint), -1)
               AND group_key = :group_key
               AND event_key like :event_key_prefix || '%'
        """

    def __init__(self, pool: DbMultihostPool):
        self._pool = pool

    @staticmethod
    def parse_event_record(rec: Record):
        data: Dict = dict(rec)
        data['context'] = ujson.loads(data['context'])
        for time_key in ['run_at', 'created_at']:
            if time_key not in data:
                continue
            data[time_key] = data[time_key].strftime('%Y-%m-%dT%H:%M:%S.%fZ')
        return data

    async def cancel(self, event_info: ExternalEventInfo, backoff_sett: BackoffSettings):
        q, p = compile_query(sa.text(self.CANCEL_Q).params(**asdict(event_info)))

        @backoff_sett.on_exception(backoff.expo, asyncpg.PostgresError)
        async def try_():
            async with self._pool().acquire(timeout=2) as conn:
                result = await conn.fetchrow(q, *p)
            err, old_status, new_state = result['err'], result['old_status'], result['new_rec']
            err_result = {'event_status': old_status}
            if err == 'not_found':
                raise EntityDoesNotExist('Event not found', fail_status=err)
            if err == 'already_cancelled':
                raise InvalidEntityCondition('Event is already cancelled', fail_status=err, fail_data=err_result)
            if err == 'illegal_status':
                raise InvalidEntityCondition(
                    f'Event can not be cancelled from current status <{old_status}>',
                    fail_status=err,
                    fail_data=err_result
                )
            if not new_state:
                raise InvalidEntityData('cancel returned no event data')
            return self.parse_event_record(new_state)

        return await try_()

    async def delete(self, event_info: ExternalEventInfo, min_run_at: datetime, force: bool, backoff_sett: BackoffSettings):
        q, p = compile_query(sa.text(self.DELETE_Q).params(**asdict(event_info), min_run_at=min_run_at, force=force))

        @backoff_sett.on_exception(backoff.expo, asyncpg.PostgresError)
        async def try_():
            async with self._pool().acquire(timeout=2) as conn:
                result = await conn.fetchrow(q, *p)
            err, state = result['err'], result['rec']
            if err == 'not_found':
                raise EntityDoesNotExist("Event not found", fail_status=err)
            if err == 'too_close_to_run':
                raise InvalidEntityCondition(
                    "Event is near it's running time, can't be deleted",
                    fail_status=err,
                    fail_data={'rec': self.parse_event_record(state)}
                )
            if not state:
                raise InvalidEntityData('delete returned no event data')
            return self.parse_event_record(state)

        return await try_()

    async def update(
            self, event_info: ExternalEventInfo, backoff_sett: BackoffSettings,
            run_at: datetime = None, cb_url: str = None, context: Any = None, ensure_pending: bool = False,
            **_
    ):
        q, p = compile_query(sa.text(self.UPDATE_Q).params(
            **asdict(event_info),
            run_at=run_at,
            cb_url=cb_url,
            context=context,
            ensure_pending=ensure_pending,
        ))

        @backoff_sett.on_exception(backoff.expo, asyncpg.PostgresError)
        async def try_():
            async with self._pool().acquire(timeout=2) as conn:
                result = await conn.fetchrow(q, *p)
            err, old_status, new_state = result['err'], result['old_status'], result['new_rec']
            err_result = {'event_status': old_status}
            if err == 'not_found':
                raise EntityDoesNotExist("Event not found", fail_status=err)
            if err == 'illegal_status':
                raise InvalidEntityCondition(
                    f'Event can not be cancelled from current status <{old_status}>',
                    fail_status=err,
                    fail_data=err_result,
                )
            if not new_state:
                raise InvalidEntityData('update returned no event data')
            return {'old_status': old_status, 'rec': self.parse_event_record(new_state)}

        return await try_()

    async def get(self, event_info: ExternalEventInfo, backoff_sett: BackoffSettings):
        q, p = compile_query(sa.text(self.GET_Q).params(**asdict(event_info)))

        @backoff_sett.on_exception(backoff.expo, asyncpg.PostgresError)
        async def try_():
            async with self._pool(ro=True).acquire(timeout=2) as conn:
                data = await conn.fetchrow(q, *p)
            if not data:
                raise EntityDoesNotExist('Event not found', fail_status='not_found')
            return self.parse_event_record(data)

        return await try_()

    async def list(self, group_key: str, client_id: int, backoff_sett: BackoffSettings):
        q, p = compile_query(sa.text(self.LIST_Q).params(group_key=group_key, client_id=client_id))

        @backoff_sett.on_exception(backoff.expo, asyncpg.PostgresError)
        async def try_():
            async with self._pool(ro=True).acquire(timeout=2) as conn:
                return list(map(self.parse_event_record, await conn.fetch(q, *p)))

        return await try_()

    async def list_by_prefix(self, group_key: str, client_id: int, event_key_prefix: str, backoff_sett: BackoffSettings):
        q, p = compile_query(sa.text(self.FIND_Q).params(group_key=group_key, client_id=client_id,
                                                         event_key_prefix=event_key_prefix))

        @backoff_sett.on_exception(backoff.expo, asyncpg.PostgresError)
        async def try_():
            return list(map(self.parse_event_record, await self._pool(ro=True).fetch(q, *p)))

        return await try_()
