import asyncio
from datetime import timedelta

import click

from aiohttp import web
from mail.python.theatre.stages.base import Stage
from mail.python.theatre.stages.db_stats.roles.director import Director, FatPollersConfig
from mail.python.theatre.stages.db_stats.types import Db, DbHost, DbSignal
from mail.python.theatre.stages.db_stats.types import TierRunPolicy
from .settings import Settings


def make_signals(settings: Settings):
    return (
        DbSignal(
            query='''
                -- poll_lost_buckets
                SELECT count(*) as lost_buckets
                  FROM reminders.buckets
                 WHERE heartbeat IS NULL
                    OR now() - heartbeat > make_interval(secs => %(heartbeat_deadline_secs)s )
            ''',
            run_every=timedelta(seconds=5),
            row_signal=False,
            sigopt_suffix='max',
            tier_run_policy=TierRunPolicy.Master,
            query_args=dict(
                heartbeat_deadline_secs=15,
            ),
        ),
        DbSignal(
            query='''
            -- poll_late_events
                SELECT count(*) as late_events
                  FROM reminders.events
                 WHERE status = 'pending'
                   AND originally_run_at < now() - make_interval(secs => %(max_legit_delay_secs)s )
            ''',
            run_every=timedelta(seconds=15),
            row_signal=False,
            sigopt_suffix='max',
            tier_run_policy=TierRunPolicy.Master,
            query_args=dict(
                max_legit_delay_secs=15,
            ),
        ),
        DbSignal(
            query='''
            -- poll_pending_events
                SELECT count(*) as pending_events
                  FROM reminders.events
                 WHERE status = 'pending'
            ''',
            run_every=timedelta(minutes=15),
            row_signal=False,
            sigopt_suffix='max',
            tier_run_policy=TierRunPolicy.Master,
        ),
    )


def make_db_stat_director(settings: Settings, env) -> Director:
    return Director(
        dbs=[
            Db(
                ctype=f'callmeback-{env.value}',
                hosts=[
                    DbHost(
                        dsn=settings.db.dsn_for_host(host),
                        geo=geo,
                    )
                    for geo, host in settings.db.hosts.items()
                ]
            )
        ],
        signals=make_signals(settings),
        db_host_status_poll_time=timedelta(seconds=30),
        fatPollersConfig=FatPollersConfig(fat_queries_poll_time=timedelta(seconds=15)),
    )


class DbStatsStage(Stage):
    def __init__(self, env: str):
        super(DbStatsStage, self).__init__(env=env, name='callmeback-dbstats', settings=Settings)

    async def create_app(self) -> web.Application:
        director = make_db_stat_director(self.settings, self.env)
        self.app.update(director=director)

        async def shutdown(_):
            await director.stop()

        await director.start()

        self.app.router.add_get('/unistat', director.unistat_handler)
        self.app.router.add_get('/ping', lambda _: web.Response(text='pong'))
        self.app.on_shutdown.append(shutdown)

        return self.app


@click.command()
@click.option('--host', default='::0', help='IP addr to bind to')
@click.option('--port', default=80, help='HTTP port to listen')
@click.option('--env', help='Environment name')
def run_main(host, port, env=None):
    import uvloop
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())

    stage = DbStatsStage(env)
    stage.run(host=host, port=port)


if __name__ == '__main__':
    run_main()
