import logging
from typing import List

import sqlalchemy as sa
from asyncpg import PostgresError
from asyncpgsa import compile_query

from mail.callmeback.callmeback.stages.worker.settings import BucketHolderSettings
from mail.python.theatre.roles import Cron
from mail.python.theatre.app.roles.db_multihost_pool import DbMultihostPool

log = logging.getLogger(__name__)


class BucketHolder(Cron):
    """Acquires buckets in DB, using heartbeat-based scheme"""
    ACQUIRE_Q = '''
        -- acquire_buckets
        WITH ready_buckets AS (
          SELECT bucket_id
            FROM reminders.buckets
           WHERE heartbeat IS NULL
                 OR now() - heartbeat > make_interval(secs => :heartbeat_deadline_secs)
                 OR hostname = :hostname
        -- Take already locked by this hostname first
        ORDER BY hostname IS NOT DISTINCT FROM :hostname DESC, bucket_id
           LIMIT :bucket_limit
           FOR NO KEY UPDATE
        )
        UPDATE reminders.buckets b
           SET hostname = :hostname,
               heartbeat = now()
          FROM ready_buckets rb
         WHERE b.bucket_id = rb.bucket_id
        RETURNING b.bucket_id
    '''

    RELEASE_Q = '''
        -- release_buckets
        UPDATE reminders.buckets
           SET heartbeat = null
         WHERE hostname = :hostname
    '''

    def __init__(self, pg_pool: DbMultihostPool, hostname: str, settings: BucketHolderSettings):
        self._bucket_ids: List[int] = []
        self._hostname = hostname
        self._pg_pool = pg_pool
        self._bucket_limit = settings.bucket_limit
        super(BucketHolder, self).__init__(job=self._routine, **settings.cron.as_dict())
        self._deadline_secs = self._run_every_sec * 3

    @property
    def bucket_ids(self):
        return self._bucket_ids

    @property
    def deadline_secs(self):
        return self._deadline_secs

    async def stop(self):
        await super(BucketHolder, self).stop()
        await self._release_buckets()

    async def _routine(self):
        try:
            await self._acquire_buckets()
        except PostgresError as e:
            log.exception(e)
            self._bucket_ids = []

    # TODO :: Should adapt tries timer on self._heartbeat
    # @backoff.on_exception(backoff.expo, PostgresError, max_tries=3, max_time=1)
    async def _acquire_buckets(self):
        q, p = compile_query(
            sa.text(self.ACQUIRE_Q).params(
                heartbeat_deadline_secs=self._deadline_secs,
                bucket_limit=self._bucket_limit,
                hostname=self._hostname,
            )
        )
        async with self._pg_pool().acquire(timeout=2) as conn:
            result = await conn.fetch(q, *p)
        self._bucket_ids = [row['bucket_id'] for row in result]

    async def _release_buckets(self):
        q, p = compile_query(sa.text(self.RELEASE_Q).params(hostname=self._hostname))
        async with self._pg_pool().acquire(timeout=2) as conn:
            await conn.fetch(q, *p)
        self._bucket_ids = []
