import datetime
import logging

import dateutil.tz
import ujson

from aiohttp import web
from aiohttp.web_response import Response
from mail.callmeback.callmeback.detail.http_helpers.tvm import TvmServiceTickets
from mail.callmeback.callmeback.stages.worker.libretto.notifier import Notifier
from mail.callmeback.callmeback.stages.worker.props.current_events import CurrentEvents
from mail.callmeback.callmeback.stages.worker.props.host_stats import HostStats
from mail.callmeback.callmeback.stages.worker.settings import Settings
from mail.python.theatre.roles import Director as DirectorBase
from mail.python.theatre.app.roles.db_multihost_pool import DbMultihostPool
from .bucket_holder import BucketHolder
from .event_marker import EventMarker
from .event_poller import EventPoller
from .herald import CallbackHerald
from .naive_schedule_planner import NaiveSchedulePlanner
from .skipped_event_finder import SkippedEventFinder

log = logging.getLogger(__name__)


class Director(DirectorBase):
    TZ = dateutil.tz.tzlocal()

    def __init__(self, pg_pool: DbMultihostPool, hostname: str, settings: Settings):
        tvm_tickets = TvmServiceTickets.from_conf(settings.tvm)
        current_events = CurrentEvents(max_events=settings.limit.max_events_in_memory)
        self.host_stats = HostStats(settings.host_stats)
        notifier = Notifier(
            tvm_tickets=tvm_tickets,
            settings=settings.notifier,
        )
        self.cb_herald = CallbackHerald(
            current_events=current_events,
            notifier=notifier,
            host_stats=self.host_stats,
            settings=settings.callback_herald,
        )
        self.buckets_holder = BucketHolder(
            pg_pool=pg_pool,
            hostname=hostname,
            settings=settings.bucket_holder,
        )
        events_marker = EventMarker(
            current_events=current_events,
            pg_pool=pg_pool,
            hostname=hostname,
            settings=settings.event_marker,
        )
        self.events_poller = EventPoller(
            pg_pool=pg_pool,
            buckets_holder=self.buckets_holder,
            herald=self.cb_herald,
            current_events=current_events,
            settings=settings.event_poller,
        )
        self.schedule_planner = NaiveSchedulePlanner(
            pg_pool=pg_pool,
            settings=settings.naive_schedule_planner,
        )
        tasks = [
            pg_pool.poller,
            notifier,
            self.cb_herald,
            self.buckets_holder,
            events_marker,
            self.events_poller,
            self.schedule_planner,
            SkippedEventFinder(events_poller=self.events_poller, settings=settings.skipped_event_finder),
        ]
        if tvm_tickets:
            tasks.append(tvm_tickets)
        super(Director, self).__init__(tasks=tasks)

    async def wakeup_handler(self, request: web.Request):
        if 'at' not in request.query:
            at = datetime.datetime.now(tz=self.TZ)
        else:
            at = datetime.datetime.utcfromtimestamp(float(request.query['at']))
        self.events_poller.wakeup_at(at)
        return 'ok'

    async def state_handler(self, request: web.Request):
        return self.events_poller.state()

    async def unistat_handler(self, _):
        cb_stats = [
            (f'callback_{host}_{stat.value}_summ', cnt)
            for host, stats in self.host_stats.counters().items() for stat, cnt in stats.items()
        ]
        bh_stats = [
            ('buckets_ammm', len(self.buckets_holder.bucket_ids))
        ]
        return Response(
            text=ujson.dumps(cb_stats + bh_stats + self.metrics()),
            content_type='application/json'
        )
