from dataclasses import asdict
from mail.callmeback.callmeback.stages.worker.props.current_events import CurrentEvents
from mail.callmeback.callmeback.stages.worker.settings import EventMarkerSettings
from mail.python.theatre.roles import Cron
from mail.python.theatre.app.roles.db_multihost_pool import DbMultihostPool


class EventMarker(Cron):
    """Marks events as notified in DB"""
    DONE_Q = '''
        -- mark_events_done
        UPDATE reminders.events
           SET status = 'notified',
               context = context || jsonb_build_object('_notify_info', jsonb_build_object(
                   'at', now(),
                   'by', $2::text,
                   'from_status', status
               ))
         WHERE event_id = ANY($1::bigint[])
    '''
    POSTPONE_Q = '''
        -- postpone_events
        WITH postpone_events AS (
            SELECT *
              FROM unnest($1::code.postpone_event_info[])
        )
        UPDATE reminders.events re
           SET run_at = now() + pe.delay,
               tries = tries + 1,
               context = context || jsonb_build_object('_postpone_info', jsonb_build_object(
                   'at', now(),
                   'by', $2::text
               ))
          FROM postpone_events pe
         WHERE re.event_id = pe.event_id
    '''

    REJECT_Q = '''
        -- reject_events
        UPDATE reminders.events
           SET status = 'rejected',
               context = context || jsonb_build_object('_reject_info', jsonb_build_object(
                   'at', now(),
                   'by', $2::text,
                   'from_status', status
               ))
         WHERE event_id = ANY($1::bigint[])
    '''

    FAIL_Q = '''
        -- fail_events
        UPDATE reminders.events
           SET status = 'failed',
               context = context || jsonb_build_object('_terminate_info', jsonb_build_object(
                   'at', now(),
                   'by', $2::text,
                   'from_status', status
               ))
         WHERE event_id = ANY($1::bigint[])
    '''

    def __init__(
            self,
            current_events: CurrentEvents,
            pg_pool: DbMultihostPool,
            hostname: str,
            settings: EventMarkerSettings,
    ):
        self._current_events = current_events
        self._pg_pool = pg_pool
        self._hostname = hostname
        super(EventMarker, self).__init__(job=self._mark_events, **settings.cron.as_dict())

    async def _mark_events(self):
        while self._current_events.done_ids:
            done_events_ids = self._current_events.done_ids[:100]
            await self.ok_events(done_events_ids)
            del self._current_events.done_ids[:len(done_events_ids)]
            for eid in done_events_ids:
                self._current_events.total_ids.remove(eid)

        while self._current_events.rejected_ids:
            rejected_events_ids = self._current_events.rejected_ids[:100]
            await self.reject_events(rejected_events_ids)
            del self._current_events.rejected_ids[:len(rejected_events_ids)]
            for eid in rejected_events_ids:
                self._current_events.total_ids.remove(eid)

        while self._current_events.failed_ids:
            failed_events_ids = self._current_events.failed_ids[:100]
            await self.fail_events(failed_events_ids)
            del self._current_events.failed_ids[:len(failed_events_ids)]
            for eid in failed_events_ids:
                self._current_events.total_ids.remove(eid)

        while self._current_events.postponed_items:
            postponed_items = self._current_events.postponed_items[:100]
            await self.postpone_events(postponed_items)
            del self._current_events.postponed_items[:len(postponed_items)]
            for item in postponed_items:
                self._current_events.total_ids.remove(item.event_id)

    async def ok_events(self, event_ids):
        async with self._pg_pool().acquire(timeout=2) as conn:
            return await conn.fetch(self.DONE_Q, event_ids, self._hostname)

    async def reject_events(self, event_ids):
        async with self._pg_pool().acquire(timeout=2) as conn:
            return await conn.fetch(self.REJECT_Q, event_ids, self._hostname)

    async def fail_events(self, event_ids):
        async with self._pg_pool().acquire(timeout=2) as conn:
            return await conn.fetch(self.FAIL_Q, event_ids, self._hostname)

    async def postpone_events(self, event_items):
        async with self._pg_pool().acquire(timeout=2) as conn:
            return await conn.fetch(self.POSTPONE_Q, [asdict(x) for x in event_items], self._hostname)
