import logging
from asyncio import TimeoutError
from datetime import datetime

import dateutil.tz
from aiohttp import ClientError, ClientResponse

from mail.callmeback.callmeback.detail.retry_params import RetryParams
from mail.callmeback.callmeback.stages.worker.props.callback_item import CallbackItem
from mail.callmeback.callmeback.stages.worker.props.current_events import CurrentEvents
from mail.callmeback.callmeback.stages.worker.props.host_stats import HostStats
from mail.callmeback.callmeback.stages.worker.settings.herald import CallbackHeraldSettings
from mail.python.theatre.roles import DelayedQueued, DelayItem
from .notifier import Notifier

log = logging.getLogger(__name__)


class BadStatusCode(RuntimeError):
    @classmethod
    async def from_resp(cls, resp: ClientResponse):
        body = await resp.text() or ""
        return cls(f'code={resp.status}, body={body}')


class CallbackHerald(DelayedQueued):
    """Delivers event notifications to end users"""
    TZ = dateutil.tz.tzlocal()

    def __init__(
            self,
            current_events: CurrentEvents,
            notifier: Notifier,
            host_stats: HostStats,
            settings: CallbackHeraldSettings,
    ):
        self._host_stats = host_stats
        self._current_events = current_events
        self._quick_retries_delay = settings.quick_retries_delay
        self._max_host_problem_coeff = settings.max_host_problem_coeff
        self._max_delayed_per_host = settings.max_delayed_per_host
        self._notifier = notifier
        self._retry_params = RetryParams(settings)
        super(CallbackHerald, self).__init__(
            job=self._process_callback,
            **settings.delayed.as_dict(),
            **settings.queued.as_dict(),
        )

    def put(self, item: CallbackItem, now: datetime = None):
        if self._current_events.set_in_progress(item.event_id):
            super(CallbackHerald, self).put(item, now)

    @staticmethod
    def notification_is_rejected(resp: ClientResponse) -> bool:
        return 'X-Ya-CallMeBack-Notify-Reject' in resp.headers

    def is_outdated(self, item: CallbackItem, now: datetime) -> bool:
        stop_after_delay = self._retry_params.get_stop_after_delay(item.retry_params)
        return item.originally_run_at + stop_after_delay < now

    async def _process_callback(self, item: CallbackItem):
        host_stats = self._host_stats[item.cb_url]
        now = datetime.now(tz=self.TZ)
        item.quick_tries = item.quick_tries + 1
        if self.is_outdated(item, now):
            log.warning('Callback is outdated, item: %r', item)
            self._current_events.fail(item.event_id)
            return
        resp: ClientResponse = None
        try:
            log.info(f'Notifying, item: {item}')
            resp = await self._notifier(item.cb_url, item)
            if self.notification_is_rejected(resp):
                self._current_events.reject(item.event_id)
                return
            if resp.status != 200:
                raise await BadStatusCode.from_resp(resp)
            self._current_events.complete(item.event_id)
            host_stats.ok(now)
            return
        except TimeoutError:
            log.warning('Notification timed out, item: %r', item)
            host_stats.timeout(now)
        except (ClientError, BadStatusCode) as e:
            log.warning('Notification failed, item: %r, exception: %r', item, e)
        if (
            (host_stats.total > self._max_delayed_per_host and host_stats.problem_coeff > self._max_host_problem_coeff)
            or (item.quick_tries > self._retry_params.get_quick_retries_count(item.retry_params))
            or (resp and resp.status == 429)
        ):
            log.warning('Postpone item: %r', item)
            host_stats.fail(now)
            delay = self._retry_params.get_retry_delay(item.retry_params, item.tries + 1)
            self._current_events.postpone(item.event_id, delay)
        else:
            delay_factor = 2 ** host_stats.problem_coeff
            raise DelayItem(now + self._quick_retries_delay * delay_factor)
