import logging
from dataclasses import asdict, dataclass, field, InitVar
from datetime import datetime, timedelta
from typing import List, Any

import dateutil.tz
import sqlalchemy as sa

import asyncpg
from asyncpgsa import compile_query
from crontab import CronTab
from mail.callmeback.callmeback.stages.worker.settings.naive_schedule_planner import NaiveSchedulePlannerSettings
from mail.python.theatre.roles import Cron
from mail.python.theatre.app.roles.db_multihost_pool import DbMultihostPool

log = logging.getLogger(__name__)


@dataclass
class ScheduleItem:
    is_active: bool
    schedule_id: int
    owner_client_id: int
    crontab: str
    cb_url: str
    context: str = field(repr=False)
    timezone: InitVar[str] = field(default='MSK')
    tz: dateutil.tz.tzfile = field(default=None, init=False)

    def __post_init__(self, timezone):
        self.tz = timezone and dateutil.tz.gettz(timezone)


# TODO :: Refactor with `api` stage
@dataclass
class NotifyItem:
    schedule_id: int
    owner_client_id: int
    group_key: str
    event_key: str
    bucket_id: int
    run_at: datetime
    cb_url: str
    context: Any


class NaiveSchedulePlanner(Cron):
    """
    Polls unplanned schedules and plan nearest events for them.
    """

    GET_SCHEDULES_Q = '''
        -- get_schedules
        SELECT sched.*
          FROM reminders.schedules sched
         WHERE is_active
           AND NOT EXISTS (
            SELECT 1
              FROM (
                SELECT *
                  FROM reminders.scheduled_events sch_ev_in
                 WHERE COALESCE(sch_ev_in.owner_client_id, -1) = COALESCE(sched.owner_client_id, -1)
                   AND sched.schedule_id = sch_ev_in.schedule_id
                 ORDER BY run_at DESC
                 LIMIT 1
              ) sch_ev
              JOIN reminders.events ev USING (event_id)
             WHERE (sch_ev.run_at > :now OR status = 'pending')
          )
          FOR NO KEY UPDATE SKIP LOCKED
    '''

    SCHEDULE_EVENTS_Q = '''
        -- schedule_events
        WITH data AS (
            SELECT *
              FROM unnest($1::code.schedule_event_info[])
        ), inserted AS (
            INSERT INTO reminders.events (
                bucket_id,
                owner_client_id,
                group_key,
                event_key,
                cb_url,
                context,
                run_at,
                originally_run_at
            )
            SELECT
                bucket_id,
                owner_client_id,
                group_key,
                event_key,
                cb_url,
                context,
                run_at,
                run_at
            FROM data
            RETURNING owner_client_id, group_key, event_key, event_id
        ) INSERT INTO reminders.scheduled_events (
            owner_client_id, run_at, schedule_id, event_id
        ) SELECT data.owner_client_id, data.run_at, data.schedule_id, inserted.event_id
            FROM data
            JOIN inserted USING (group_key, event_key)
           WHERE data.owner_client_id IS NOT DISTINCT FROM inserted.owner_client_id
    '''

    TZ = dateutil.tz.tzlocal()

    def __init__(
        self,
        pg_pool: DbMultihostPool,
        settings: NaiveSchedulePlannerSettings,
    ):
        self._pg_pool = pg_pool
        self._settings = settings

        super().__init__(job=self.schedule_events, **settings.cron.as_dict())

    async def schedule_events(self):
        now = datetime.now(tz=self.TZ)
        async with self._pg_pool().acquire(timeout=2) as conn:
            async with conn.transaction():
                items = [x async for x in self._events_to_schedule(now=now, conn=conn)]
                # TODO :: remove this when smart per-bucket event id processing is implemented
                await conn.execute('select pg_advisory_xact_lock(42)')
                return await conn.execute(self.SCHEDULE_EVENTS_Q, [asdict(x) for x in items])

    async def _events_to_schedule(self, now: datetime, conn: asyncpg.Connection):
        for sched in await self._get_schedules(now=now, conn=conn):
            try:
                delay = timedelta(
                    seconds=CronTab(crontab=sched.crontab).next(
                        now=now.astimezone(sched.tz)
                    )
                )
            except ValueError as e:
                log.error("Can't schedule event for schedule %r", sched)
                log.exception(e)
                continue
            run_at = now + delay
            yield NotifyItem(
                schedule_id=sched.schedule_id,
                owner_client_id=sched.owner_client_id,
                cb_url=sched.cb_url,
                context=sched.context,
                # TODO :: reuse worker choose strategy from `api` stage
                bucket_id=1,
                group_key=f'_schedule_{sched.schedule_id}',
                event_key=f'{run_at.isoformat(timespec="seconds")}',
                run_at=run_at,
            )

    async def _get_schedules(self, now: datetime, conn: asyncpg.Connection) -> List[ScheduleItem]:
        q, p = compile_query(
            sa.text(self.GET_SCHEDULES_Q).params(
                now=now,
            )
        )
        return [ScheduleItem(**rec) for rec in await conn.fetch(q, *p)]
