import logging

from aiohttp import ClientResponse
from contextlib import asynccontextmanager

from mail.callmeback.callmeback.detail.http_helpers.tvm import TvmServiceTickets
from mail.callmeback.callmeback.stages.worker.props.callback_item import CallbackItem
from mail.callmeback.callmeback.stages.worker.settings import NotifierSettings
from mail.python.theatre.roles import HttpPooled

log = logging.getLogger(__name__)


class Notifier(HttpPooled):
    def __init__(self, tvm_tickets: TvmServiceTickets, settings: NotifierSettings):
        self._tvm_tickets = tvm_tickets
        self.__ssl = settings.ssl
        super(Notifier, self).__init__(**settings.http_pool.as_dict())

    async def __call__(self, cb_url, item: CallbackItem) -> ClientResponse:
        async with self._notify_http(cb_url, item) as resp:
            log.debug(resp)
            return resp

    @asynccontextmanager
    async def _notify_http(self, cb_url, item: CallbackItem) -> ClientResponse:
        http_context = item.context.get('_http', {})
        headers: dict = http_context.get('headers', {})
        if self._tvm_tickets and item.owner_client_id:
            headers['X-Ya-Service-Ticket'] = await self._tvm_tickets.get(item.owner_client_id)
        if http_context.get('add_callmeback_headers'):
            headers.update((k, v) for k, v in {
                'X-Ya-CallMeBack-Group-Key': str(item.group_key),
                'X-Ya-CallMeBack-Event-Key': str(item.event_key),
            }.items() if v)
        async with self.http_session.request(
                method=http_context.get('method', 'get'),
                url=cb_url,
                params=http_context.get('params'),
                data=http_context.get('data'),
                json=http_context.get('json'),
                headers=headers,
                ssl=self.__ssl,
        ) as resp:
            yield resp
