import asyncio
import os
import pathlib

import click
import ujson
from aiohttp import web
from datetime import timedelta

from mail.callmeback.callmeback.detail.http_helpers.middleware import jsend_middleware
from mail.callmeback.callmeback.stages.worker.libretto.director import Director
from mail.python.theatre.app.roles.db_multihost_pool import DbMultihostPool
from mail.python.theatre.stages.base import Stage
from .settings import Settings


class WorkerStage(Stage):
    def __init__(self, env: str, hostname: str):
        super(WorkerStage, self).__init__(env=env, name='callmeback-worker', settings=Settings)
        self.hostname = hostname

    async def create_app(self):
        pg_pool = await DbMultihostPool(
            db_settings=self.settings.db,
            db_host_status_poll_time=timedelta(seconds=30)
        ).create_pool()

        director = Director(pg_pool=pg_pool, hostname=self.hostname, settings=self.settings)
        self.app.update(pool=pg_pool, director=director)
        await director.start()

        async def shutdown(_):
            await director.stop()
            await pg_pool.close()

        self.app.on_shutdown.append(shutdown)
        self.setup_routes(director)
        return self.app

    def setup_routes(self, proc):
        v1app = web.Application(middlewares=[jsend_middleware])
        v1app.router.add_get('/wakeup', proc.wakeup_handler, name='wakeup')
        v1app.router.add_get('/getstate', proc.state_handler, name='state')
        self.app.add_subapp('/v1', v1app)

        self.app.router.add_get('/unistat', proc.unistat_handler, name='unistat')
        self.app.router.add_get('/ping', lambda _: web.Response(text='pong'))


def get_hostname(port):
    host = 'localhost'
    qloud_conf = pathlib.Path('/etc/qloud/meta.json')
    if qloud_conf.exists():
        with open(str(qloud_conf)) as fd:
            host = ujson.load(fd)['hostname']
    return os.environ.get('CALLMEBACK_WORKER_URL', f'http://{host}:{port}/v1/wakeup')


@click.command()
@click.option('--host', default='::0', help='IP addr to bind to')
@click.option('--port', default=80, help='HTTP port to listen')
@click.option('--env', help='Environment name')
def run_main(host, port, env=None):
    import uvloop
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())

    stage = WorkerStage(env=env, hostname=get_hostname(port))
    stage.run(host=host, port=port)


if __name__ == '__main__':
    run_main()
