import logging
from typing import List, Set
from datetime import timedelta

from dataclasses import dataclass, field

log = logging.getLogger(__name__)


@dataclass
class PostponeItem:
    event_id: int
    delay: timedelta


@dataclass
class CurrentEvents:
    """Info about partially processed events"""
    max_events: int
    in_progress_ids: Set[int] = field(default_factory=set)
    done_ids: List[int] = field(default_factory=list)
    failed_ids: List[int] = field(default_factory=list)
    postponed_items: List[PostponeItem] = field(default_factory=list)
    rejected_ids: List[int] = field(default_factory=list)
    total_ids: Set[int] = field(default_factory=set)

    def set_in_progress(self, event_id: int):
        if event_id in self.in_progress_ids:
            log.warning('event_id=%d is already in progress', event_id)
            return False
        log.info('event_id=%d state change: in_progress', event_id)
        self.in_progress_ids.add(event_id)
        self.total_ids.add(event_id)
        return True

    def complete(self, event_id: int):
        return self._progress_to(self.done_ids, event_id, 'complete')

    def fail(self, event_id: int):
        return self._progress_to(self.failed_ids, event_id, 'fail')

    def postpone(self, event_id: int, delay: timedelta):
        if event_id not in self.in_progress_ids:
            log.warning('event_id=%d is not in progress', event_id)
            return False
        log.info('event_id=%d state change: postpone', event_id)
        self.in_progress_ids.remove(event_id)
        self.postponed_items.append(PostponeItem(event_id, delay))
        return True

    def reject(self, event_id: int):
        return self._progress_to(self.rejected_ids, event_id, 'reject')

    def _progress_to(self, collection: List[int], event_id: int, to_state: str):
        if event_id not in self.in_progress_ids:
            log.warning('event_id=%d is not in progress', event_id)
            return False
        log.info('event_id=%d state change: %s', event_id, to_state)
        self.in_progress_ids.remove(event_id)
        collection.append(event_id)
        return True

    @property
    def available_events_cnt(self):
        return self.max_events - len(self.total_ids)
