from array import array
from collections import defaultdict
from datetime import timedelta, datetime
from enum import Enum, unique
from typing import Dict, Union

import dateutil.tz
from cachetools.func import lfu_cache
from yarl import URL

from mail.callmeback.callmeback.stages.worker.settings import HostStatsSettings


class WindowCounter:
    def __init__(self, window_size: timedelta, buckets_cnt: int, now: datetime):
        self._bucket_size = window_size / buckets_cnt
        self._buckets = array('I', [0] * buckets_cnt)
        self._bucket_idx = 0
        self._window_end = now + self._bucket_size
        self._total = 0

    def inc(self, now: datetime):
        # Count of buckets to be cleared
        stale_buckets_cnt = min(
            int((now - self._window_end) / self._bucket_size) + 1,
            len(self._buckets),
        )
        for _ in range(stale_buckets_cnt):
            # Move pointer to current bucket
            self._bucket_idx = (self._bucket_idx + 1) % len(self._buckets)
            self._window_end += self._bucket_size
            # Clear current bucket
            self._total -= self._buckets[self._bucket_idx]
            self._buckets[self._bucket_idx] = 0

        self._buckets[self._bucket_idx] += 1
        self._total += 1

    @property
    def total(self):
        return self._total

    @property
    def buckets(self):
        return self._buckets[self._bucket_idx:] + self._buckets[:self._bucket_idx]


@unique
class WindowKind(Enum):
    OKS = 'oks'
    FAILS = 'fails'
    TIMEOUTS = 'timeouts'


class HostWindowStat:
    TZ = dateutil.tz.tzlocal()
    COEFFS: Dict[WindowKind, float] = {
        WindowKind.OKS: 0,
        WindowKind.FAILS: 1,
        WindowKind.TIMEOUTS: 5,
    }

    def __init__(self, window_size: timedelta, buckets_cnt: int):
        self.window_size = window_size
        now = datetime.now(tz=self.TZ)
        self.windows: Dict[WindowKind, WindowCounter] = {
            kind: WindowCounter(window_size, buckets_cnt, now) for kind in WindowKind
        }
        self.cntrs: Dict[WindowKind, int] = defaultdict(int)

    def ok(self, now: datetime):
        self.windows[WindowKind.OKS].inc(now)
        self.cntrs[WindowKind.OKS] += 1

    def fail(self, now: datetime):
        self.windows[WindowKind.FAILS].inc(now)
        self.cntrs[WindowKind.FAILS] += 1

    def timeout(self, now: datetime):
        self.windows[WindowKind.TIMEOUTS].inc(now)
        self.cntrs[WindowKind.TIMEOUTS] += 1

    @property
    def total(self):
        return sum(w.total for w in self.windows.values())

    @property
    def problem_coeff(self):
        return (
            (sum(w.total * self.COEFFS[kind] for kind, w in self.windows.items()) + 1)
            /
            (self.total + 1)
        )


class HostStats:
    def __init__(self, settings: HostStatsSettings):
        self._hosts: Dict[str, HostWindowStat] = defaultdict(
            lambda: HostWindowStat(settings.window_size, settings.buckets_cnt)
        )

    def __getitem__(self, url: Union[str, URL]):
        if isinstance(url, str):
            url = self.url(url)
        return self._hosts[url.host]

    def counters(self) -> Dict[str, Dict[WindowKind, int]]:
        return {host: stat.cntrs for host, stat in self._hosts.items()}

    @lfu_cache(maxsize=10*1024)
    def url(self, host: str) -> URL:
        return URL(host)
