create schema reminders;

CREATE TYPE reminders.event_status AS ENUM (
    'pending',
    'notified',
    'failed',
    'cancelled',
    'rejected'
);

CREATE TABLE reminders.buckets (
    bucket_id  integer not null PRIMARY KEY,
    hostname   text,
    heartbeat  timestamptz
);

CREATE TABLE reminders.events (
    -- PK
    event_id   bigserial primary key,
    -- Microshard FK
    bucket_id  integer,
    -- External keys
    event_key  text,
    group_key  text,
    -- Payload data
    run_at     timestamptz,
    context    jsonb,
    cb_url     text,
    -- Internal data
    owner_client_id bigint,
    created_at timestamptz not null default now(),
    status     reminders.event_status not null default 'pending',
    originally_run_at timestamptz not null,
    tries      integer NOT NULL DEFAULT 0,

    CONSTRAINT fk_events_to_buckets FOREIGN KEY (bucket_id)
        REFERENCES reminders.buckets(bucket_id)
        ON DELETE RESTRICT
);

CREATE INDEX ix_events_by_bucket_id_run_at ON reminders.events(bucket_id, run_at, event_id)
    WHERE status = 'pending';

CREATE INDEX ix_events_processed_by_run_at ON reminders.events(run_at, event_id)
    WHERE status != 'pending';

CREATE UNIQUE INDEX ux_events_by_keys_new ON reminders.events(COALESCE(owner_client_id, -1), group_key, event_key)
    WHERE group_key IS NOT NULL;


CREATE TYPE reminders.change_type AS ENUM (
  'delete',
  'update'
);


CREATE TABLE reminders.change_log (
  cid bigserial primary key,
  at timestamptz default now(),
  event_id bigint,
  owner_client_id bigint,
  event_key  text,
  group_key  text,
  change_type reminders.change_type,
  change_info jsonb,
  event_data reminders.events
);

CREATE INDEX i_change_log_by_event_id ON reminders.change_log (event_id);
CREATE INDEX i_change_log_by_external_keys ON reminders.change_log (owner_client_id, group_key, event_key);
CREATE INDEX i_change_log_by_group_key ON reminders.change_log (owner_client_id, group_key, at);
CREATE INDEX i_change_log_by_at ON reminders.change_log (at, cid);


CREATE TABLE reminders.schedules (
    schedule_id     bigserial primary key,
    owner_client_id bigint,
    is_active       boolean default true,
    crontab         text not null,
    -- Payload
    context jsonb not null,
    cb_url  text not null
); 

CREATE INDEX i_schedules_by_owner_client_id_new ON reminders.schedules (COALESCE(owner_client_id, -1), schedule_id);


CREATE TABLE reminders.scheduled_events (
    -- Owner_client_id for records locality @ indexes
    owner_client_id    bigint,
    run_at timestamptz not null,
    schedule_id bigint not null,
    event_id bigint    not null,

    CONSTRAINT fk_scheduled_events_to_events FOREIGN KEY (event_id)
        REFERENCES reminders.events
        ON DELETE CASCADE
);

CREATE UNIQUE INDEX ux_scheduled_events_new ON reminders.scheduled_events (COALESCE(owner_client_id, -1), schedule_id, run_at);
CREATE UNIQUE INDEX ux_scheduled_events_event_id ON reminders.scheduled_events (event_id);
