CREATE TYPE code.update_event_error AS ENUM (
    'not_found',
    'illegal_status'
);

CREATE TYPE code.update_event_result as (
    err code.update_event_error,
    old_status reminders.event_status,
    new_rec reminders.events
);

create or replace function code.update_event(
    i_owner_client_id int,
    i_group_key text,
    i_event_key text,
    i_run_at timestamptz,
    i_cb_url text,
    i_context jsonb,
    i_ensure_pending boolean default false
) RETURNS code.update_event_result as $$
DECLARE
  v_rec reminders.events;
  v_new_rec reminders.events;
BEGIN
  SELECT *
    INTO v_rec
    FROM reminders.events
   WHERE COALESCE(owner_client_id, -1) = COALESCE(i_owner_client_id, -1)
     AND group_key = i_group_key
     AND event_key = i_event_key
  FOR NO KEY UPDATE;

  if not found then
    return ('not_found', null, null)::code.update_event_result;
  end if;

  if (
      i_ensure_pending and v_rec.status not in ('cancelled', 'pending')
  ) or (
      not i_ensure_pending and v_rec.status != 'pending'
  ) then
    return ('illegal_status', v_rec.status, null)::code.update_event_result;
  end if;

  UPDATE reminders.events
     SET status = CASE WHEN i_ensure_pending THEN 'pending' ELSE status END,
         run_at = COALESCE(i_run_at, run_at),
         originally_run_at = COALESCE(i_run_at, originally_run_at),
         cb_url = COALESCE(i_cb_url, cb_url),
         context = COALESCE(i_context, context)
   WHERE event_id = v_rec.event_id
  RETURNING *
  INTO v_new_rec;

  INSERT INTO reminders.change_log (
    event_id, owner_client_id, event_key, group_key, change_type, event_data,
    change_info
    )
  VALUES (
    v_rec.event_id, v_rec.owner_client_id, v_rec.event_key, v_rec.group_key, 'update', v_rec,
    jsonb_build_object('run_at', i_run_at, 'cb_url', i_cb_url, 'context', i_context, 'ensure_pending', i_ensure_pending)
  );

  return (null, v_rec.status, v_new_rec)::code.update_event_result;
END;
$$ language plpgsql;