#include <mail/catdog/address_parser/lib/address_parser.h>

#include <butil/network/rfc2822.h>
#include <butil/email/email.h>
#include <butil/email/yandex_login.h>
#include <butil/network/domain_check.h>
#include <butil/StrUtils/utfutils.h>
#include <butil/StrUtils/pystrutils.h>

#include <sstream>

namespace address_parser {

std::string toString(const Recipient& r) {
    std::ostringstream out;
    out << r;
    return out.str();
}

Recipients parseRecipients(const std::string& recipients) {
    Recipients emails;
    try {
        rfc2822ns::address_iterator it(recipients), end;

        while (it != end) {
            std::string domain = it.domain();
            std::string local = it.local();

            utf::to_lower(local);
            utf::to_lower(domain);

            const bool invalidAddress = local.empty() || domain.empty();
            if (invalidAddress) {
                emails.emplace_back(Recipient{"", "", "", recipients, false});
                break;
            }

            const bool invalid = invalidAddress || local.find('%') != std::string::npos ||
                                 (is_yandex_host(domain) && !isValidYandexLogin(local));

            const auto display = pystrutils::get_validated_utf8(it.display());

            it++;
            emails.emplace_back(Recipient{display, local, domain,
                                          rfc2822ns::join_address(display, local, domain), !invalid});

        }
    } catch (const std::exception&) {
        emails.emplace_back(Recipient{"", "", "", recipients, false});
    }

    return emails;
}

std::ostream& operator<<(std::ostream& out, const Recipient& r) {
    out << "display_name: '" << r.display_name << "' "
        << "local: '" << r.local << "' "
        << "domain: '" << r.domain << "' "
        << "email: '" << r.email << "' "
        << "valid: " << std::boolalpha << r.valid;

    return out;
}

bool operator==(const Recipient& r1, const Recipient& r2) {
    return r1.display_name == r2.display_name &&
           r1.local == r2.local &&
           r1.domain == r2.domain &&
           r1.email == r2.email &&
           r1.valid == r2.valid;
}

}
