import argparse
import asyncio
import sys
import os

import uvloop
from aiohttp import web

from mail.catdog.catdog.handlers.emails import handle_emails
from mail.catdog.catdog.handlers.stat import handle_stat
from mail.catdog.catdog.src import stat, tvm
from mail.catdog.catdog.src import logger
from mail.catdog.catdog.src import config
from mail.catdog.catdog.src.config import cfg
from mail.catdog.catdog.src import http_client
from mail.catdog.catdog.src.yasrv import load as loadYasrv
from mail.catdog.catdog.src.top200 import load as loadTop200
from mail.catdog.catdog.src.staff import load as loadStaff
from mail.catdog.catdog.src.s3 import S3
from cpp import loadDomains, loadPalette


def get_port_and_config(argv):
    parser = argparse.ArgumentParser(description='Run the CatDog service on the selected port')
    parser.add_argument('--port', type=int, help='port for running CatDog service', required=True)
    parser.add_argument('config', type=str, help='path to config')

    args = parser.parse_args(args=argv)

    return args.port, args.config


def load_lists(s3):
    loadPalette(cfg().palette_path)
    loadDomains(cfg().pdd_domains)
    loadYasrv(s3)
    loadTop200(s3)
    loadStaff(s3)


async def handle_reload_lists(s3):
    try:
        load_lists(s3)
        return web.Response(text='lists reloaded', status=200)
    except Exception as e:
        return web.Response(text='{0}: {1}'.format(type(e).__name__, str(e)), status=500)


def main():
    argv = sys.argv[1:]
    port, path_to_cfg = get_port_and_config(argv)
    stat.instance = stat.Stat(os.getenv('QLOUD_INSTANCE', 'localhost') + '_' + str(port))
    config.init(path_to_cfg, os.getenv('ENVIRONMENT_NAME'), os.getenv('ENVIRONMENT_TYPE'))
    tvm.init(config.instance.tvm2, config.instance.env)
    s3 = S3(config.instance.s3)

    with logger.Holder(str(port)) as log:
        logger.log(level='info', message='catdog started', reason="for god's sake", port=port)

        load_lists(s3)

        asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())

        app = web.Application()
        asyncio.get_event_loop().run_until_complete(http_client.set_session())

        app.router.add_get('/ping', lambda _: web.Response(text='pong\n'))
        app.router.add_get('/reload', lambda _: web.Response(text='reloaded\n') if log.reopen()
                           else web.Response(text='not reloaded\n'))
        app.router.add_get('/reload_lists', lambda _: handle_reload_lists(s3))
        app.router.add_get('/stat', handle_stat)
        app.router.add_post('/emails', handle_emails)

        web.run_app(app, host='::1', port=port)


if __name__ == '__main__':
    main()
