import os
import requests

from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import SubprocessComponent, WithPort
from library.python import resource


class CatdogTesting(SubprocessComponent, WithPort):
    NAME = 'catdog_testing'
    DEPS = []

    def get_etc_path(self):
        return os.path.join(self.root, 'etc', 'catdog')

    def get_log_dir(self):
        return os.path.join(self.root, 'var', 'log', 'catdog')

    def __init__(self, env, components):
        super().__init__(env, components)
        self.bin_file = env.get_arcadia_bin('mail/catdog/catdog/bin/catdog')
        self.config_path = os.path.join(self.etc_path, 'catdog.yaml')
        self.s3_credentials_path = os.path.join(self.root, 's3_credentials')

    def init_root(self):
        helpers.mkdir_recursive(self.log_dir)
        helpers.mkdir_recursive(self.etc_path)

        helpers.write2file(resource.find('s3_credentials'), self.s3_credentials_path)
        helpers.write2file(resource.find('tvm_secret'), os.path.join(self.etc_path, 'tvm_secret'))

        for cfg in ['catdog.yaml', 'domains.txt', 'palette.conf', 'pdd-domains.txt']:
            helpers.write2file(resource.find(os.path.join('etc', cfg)), os.path.join(self.etc_path, cfg))

    def start(self):
        cmd = [self.bin_file, self.config_path, '--port', str(self.port)]
        env = {
            'ENVIRONMENT_TYPE': 'development',
            'AWS_SHARED_CREDENTIALS_FILE': self.s3_credentials_path
        }

        with self.starting(cmd=cmd, env=env):
            helpers.wait_ping(self.logger, self.healthcheck)

    def healthcheck(self, timeout=1):
        url = f'http://localhost:{self.port}/ping'
        self.logger.info('GET request, url: %s', url)
        return requests.get(url, timeout=timeout)

    def info(self):
        return {
            'bin_file': self.bin_file,
            'port': self.port,
            **super().info()
        }
