import asyncio
import collections
import json
import time


from aiohttp import web

from mail.catdog.catdog.src import error
from mail.catdog.catdog.src import parser
from mail.catdog.catdog.src import logic
from mail.catdog.catdog.src import stat

from mail.catdog.catdog.handlers import context

Task = collections.namedtuple('task', ['raw', 'email', 'coro'])


def handle_single_raw(raw, ctx):
    tasks = []
    ret = []
    for email in parser.parse_email(raw, ctx):
        if email.valid:
            h = logic.choose_handlers(email, ctx)

            res = h.process(email, ctx)
            if asyncio.iscoroutine(res):
                tasks.append(Task(raw=raw, email=email, coro=res))
            else:
                ret.append(res.as_dict())
        else:
            ctx.log_debug(reason='handle_single_email', message='invalid email', email=email.email)
            stat.instance.increase('invalid')
            ret.append(email.as_dict())

    return tasks, ret


async def handle_emails(req):
    ctx = context.Context(req)
    ctx.log_info(reason='handle_emails', message='accept', ip=ctx.ip, uid=ctx.uid)
    now = time.time()
    try:
        vec = await req.json()
        ctx.log_debug(reason='handle_emails', message='access_body', body=vec)

        async_tasks = []
        results = collections.defaultdict(list)
        for raw in vec:
            tasks, ret = handle_single_raw(raw, ctx)
            results[raw] = ret
            async_tasks += tasks

        if len(async_tasks) != 0:
            for raw, email in zip([t.raw for t in async_tasks],
                                  await asyncio.gather(*[t.coro for t in async_tasks])):
                results[raw].append(email.as_dict())

        status = 200
        text = json.dumps(results)
    except (json.JSONDecodeError, TypeError) as e:
        ctx.log_error(reason='handle_emails json decode error', message=str(e))
        status = 400
        text = error.make(e)
    except Exception as e:
        ctx.log_error(reason='handle_emails internal error', message=str(e))
        status = 500
        text = error.make(e)
    resp = web.Response(text=text, status=status, content_type='application/json')
    ctx.log_info(reason='handle_emails', message='done', status=status, time='{:1.6f}'.format(time.time() - now))
    return resp
